jls.loader.provide('jls.io.cs.Charset');

jls.loader.require('jls.io.cs.CharDecoder');
jls.loader.require('jls.io.cs.CharEncoder');

/**
 * @namespace Provides character sets.
 * @see jls.io.cs.Charset
 * @name jls.io.cs
 */

/**
 * @class This class represents a character set and the associated codec.
 * The default charsets are Cp1252, Cp850, ISO-8859-1, UTF-8 and ASCII.
 */
jls.io.cs.Charset = jls.lang.Class.create( /** @lends jls.io.cs.Charset.prototype */
{
    /**
     * Creates a charset.
     * 
     * @private
     */
    initialize : function(name) {
        this._name = name;
    },
    /**
     * Returns the charset name.
     *
     * @returns {String} The charset name.
     */
    getName : function() {
        return this._name;
    },
    /**
     * Decodes byte buffer and returns the string.
     *
     * @param {jls.lang.Buffer} buf The buffer to decode.
     * @returns {String} The decoded string.
     */
    decode : function(buf) {
        return '';
    },
    /**
     * Encodes a string and returns a buffer.
     *
     * @param {String} str The string to encode.
     * @returns {jls.lang.Buffer} The buffer.
     */
    encode : function(str) {
        return null;
    },
    /**
     * Creates a new decoder for this charset.
     *
     * @returns {jls.io.cs.CharDecoder} The buffer.
     */
    newDecoder : function() {
        return new jls.io.cs.CharDecoder(this);
    },
    /**
     * Creates a new encoder for this charset.
     *
     * @returns {jls.io.cs.CharEncoder} The buffer.
     */
    newEncoder : function() {
        return new jls.io.cs.CharEncoder(this);
    }
});


Object.extend(jls.io.cs.Charset, /** @lends jls.io.cs.Charset */
{
	_defaultCharset : null,
	_charsetMapping : {
		'Cp1252' : 'jls.io.cs.Cp1252',
		'Cp850' : 'jls.io.cs.Cp850',
		'ISO-8859-1' : 'jls.io.cs.ISO_8859_1',
		'UTF-8' : 'jls.io.cs.UTF_8',
		'ASCII' : 'jls.io.cs.Charset'
	},
	_availableCharsets : {},
    /**
     * Returns available charsets.
     * 
     * @returns {Array} The available charsets.
     */
	availableCharsets : function() {
        return Object.values(jls.io.cs.Charset._availableCharsets);
    },
	addCharset : function(cs) {
		jls.io.cs.Charset._availableCharsets[cs.getName()] = cs;
    },
    /**
     * Returns the default charset.
     * 
     * @returns {jls.io.cs.Charset} The default charset.
     */
    defaultCharset : function() {
    	if (jls.io.cs.Charset._defaultCharset == null) {
            var csn = _native.core.properties['file.encoding'];
    		if (csn && jls.io.cs.Charset.isSupported(csn)) {
    			jls.io.cs.Charset._defaultCharset = jls.io.cs.Charset.forName(csn);
    		} else {
        		jls.io.cs.Charset._defaultCharset = jls.io.cs.Charset.forName('ASCII');
    		}
    	}
    	return jls.io.cs.Charset._defaultCharset;
    },
    /**
     * Tells if the specified charset is supported.
     * 
     * @param {String} name The charset name.
     * @returns {Boolean} true if the specified charset is supported.
     */
    isSupported : function(name) {
    	if (typeof name != 'string') {
			throw new jls.lang.Exception('Invalid charset name');
    	}
    	if (name in jls.io.cs.Charset._availableCharsets) {
			return true;
    	}
    	if (name in jls.io.cs.Charset._charsetMapping) {
            jls.loader.require(jls.io.cs.Charset._charsetMapping[name]);
			return name in jls.io.cs.Charset._availableCharsets;
    	}
		return false;
    },
    /**
     * Returns the specified charset.
     * 
     * @param {String} name The charset name.
     * @returns {jls.io.cs.Charset} The charset.
     */
	forName : function(name) {
		if (! jls.io.cs.Charset.isSupported(name)) {
			throw new jls.lang.Exception('Unsupported charset "' + name + '"');
		}
        return jls.io.cs.Charset._availableCharsets[name];
    }
});

// static
jls.io.cs.Charset.addCharset(new jls.io.cs.Charset('ASCII'));


