jls.loader.provide('jls.security.MessageDigest');

jls.loader.require('jls.lang.ByteBuffer');
jls.loader.require('jls.io.BufferChannel');

/**
 * @namespace Provides security classes for the jls language.
 * @see jls.security.MessageDigest
 * @name jls.security
 */

/**
 * @class This class represents a cryptographic hash function.
 * The message digests are MD5(not implemented) and SHA1.
 */
jls.security.MessageDigest = jls.lang.Class.create( /** @lends jls.security.MessageDigest.prototype */
{
    /**
     * Creates a MessageDigest.
     * 
     * @private
     */
    initialize : function(algorithm) {
        this._algorithm = algorithm;
        this._buffer = jls.lang.ByteBuffer.allocate(1024);
    },
    /**
     * Returns the algorithm name.
     *
     * @returns {String} The algorithm name.
     */
    getAlgorithm : function() {
        return this._algorithm;
    },
    /**
     * Completes the digest and returns a buffer.
     *
     * @returns {jls.lang.ByteBuffer} The digested buffer.
     */
    digest : function() {
        throw new jls.lang.Exception('Not implemented');
    },
    /**
     * Resets the digest.
     *
     */
    reset : function() {
        this._buffer.clear();
    },
    _checkRemaining : function(length) {
        if (this._buffer.remaining() >= length) {
            return;
        }
        var capacity = this._buffer.capacity() * 2;
        for (; length > capacity - this._buffer.capacity(); capacity = capacity * 2);
        var tmp = jls.lang.ByteBuffer.allocate(capacity);
        this._buffer.flip();
        tmp.putBuffer(this._buffer);
        this._buffer = tmp;
    },
    /**
     * Updates the digest with a specified buffer.
     *
     * @param {jls.lang.ByteBuffer} buffer The buffer to update.
     */
    update : function(buffer) {
        this._checkRemaining(buffer.remaining());
        this._buffer.putBuffer(buffer, buffer.remaining());
    },
    /**
     * Updates the digest with a specified byte.
     *
     * @param {Number} b The byte to update.
     */
    updateByte : function(b) {
        this._checkRemaining(1);
        this._buffer.putByte(b);
    },
    /**
     * Updates the digest with a specified string.
     *
     * @param {String} buffer The string to update.
     */
    updateString : function(s, csn) {
        this._checkRemaining(s.length * 2);
        this._buffer.putString(s, csn ? csn : 'UTF-8');
    }
});

Object.extend(jls.security.MessageDigest, /** @lends jls.security.MessageDigest */
{
	_algorithmMapping : {
		'SHA1' : 'jls.security.Sha1'
	},
	_availableMessageDigests : {},
    /**
     * Returns available algorithms.
     * 
     * @returns {Array} The available algorithms.
     */
	availableAlgorithms : function() {
        return Object.keys(jls.security.MessageDigest._availableMessageDigests);
    },
	addMessageDigest : function(mdc) {
    	if ((typeof mdc == 'undefined') || (mdc == null)) {
			throw new jls.lang.Exception('Invalid message digest argument');
    	}
        var md = new mdc();
    	if (! (md instanceof jls.security.MessageDigest)) {
			throw new jls.lang.Exception('Invalid message digest argument (not an instance of MessageDigest)');
    	}
		jls.security.MessageDigest._availableMessageDigests[md.getAlgorithm()] = mdc;
    },
    /**
     * Tells if the specified algorithm is supported.
     * 
     * @param {String} algorithm The algorithm name.
     * @returns {Boolean} true if the specified algorithm is supported.
     */
    isSupported : function(algorithm) {
    	if (typeof algorithm != 'string') {
			throw new jls.lang.Exception('Invalid algorithm name');
    	}
    	if (algorithm in jls.security.MessageDigest._availableMessageDigests) {
			return true;
    	}
    	if (algorithm in jls.security.MessageDigest._algorithmMapping) {
            jls.loader.require(jls.security.MessageDigest._algorithmMapping[algorithm]);
			return algorithm in jls.security.MessageDigest._availableMessageDigests;
    	}
		return false;
    },
    /**
     * Returns the specified message digest.
     * 
     * @param {String} algorithm The message digest algorithm name.
     * @returns {jls.security.MessageDigest} The message digest.
     */
	getInstance : function(algorithm) {
		if (! jls.security.MessageDigest.isSupported(algorithm)) {
			throw new jls.lang.Exception('Unsupported algorithm "' + algorithm + '"');
		}
        return new jls.security.MessageDigest._availableMessageDigests[algorithm]();
    }
});
