/*!
 * 
 * Copyright (c) 2009 - 2011, javalikescript@free.fr (SPYL). All rights reserved.
 * JLS is licensed under LGPL, see the license.txt file that accompanied this code.
 * 
 */
/**
 * jls Base framework.
 * All the framework objects are created under the jls namespace.
 * The default objects are extended using prototype like behaviors.
 * 
 * @namespace
 * @see jls.lang
 */
var jls = {};

/**
 * @namespace Provides base classes for the jls language.
 * @see jls.lang.Exception
 * @see jls.lang.Logger
 * @see jls.lang.ClassLoader
 */
jls.lang = {};

// Load prototype light
_native.core.evalScript('plight.js');

// Load base classes
_native.core.evalScript('jls/lang/Exception.js');
_native.core.evalScript('jls/lang/Logger.js');

/**
 * Logger.
 * 
 * @type jls.lang.Logger
 * @memberOf jls
 */
jls.logger = new jls.lang.Logger(jls.lang.Logger.WARN);

if ('jls.logger.logLevel' in _native.core.properties) {
    jls.logger.setLogLevel(_native.core.properties['jls.logger.logLevel']);
}

_native.core.evalScript('jls/lang/ClassLoader.js');

/**
 * ClassLoader.
 * 
 * @type jls.lang.ClassLoader
 * @memberOf jls
 */
jls.loader = new jls.lang.ClassLoader();

// Register base classes
jls.loader.provide('jls.lang.Exception', true);
jls.loader.provide('jls.lang.Logger', true);
jls.loader.provide('jls.lang.ClassLoader', true);

/*
 * Provide all lang classes
 */
jls.loader.require('jls.lang.Buffer');
jls.loader.require('jls.lang.Process');
jls.loader.require('jls.lang.Runtime');
jls.loader.require('jls.lang.System');

if (! ('gui.toolkit' in _native.core.properties)) {
	_native.core.properties['gui.toolkit'] = _native.core.properties['os.sysname'] == 'Windows_NT' ? 'win32' : 'gui';
}

jls.lang.System._initialize();

jls.loader.require('jls.lang.Thread');
jls.loader.require('jls.lang.Signal');
jls.loader.require('jls.lang.Lock');
jls.loader.require('jls.lang.Monitor');
//jls.loader.require('jls.lang.ProcessBuilder');
jls.loader.require('jls.lang.Struct');
//jls.loader.require('jls.lang.AssertionError');

if ('jls.disableDefaultExceptionHandler' in _native.core.properties) {
    jls.logger.info('Default exception handler disabled');
} else {
    /*
     * Override exception handler
     */
    _native.core.exceptionHandler = function(e) {
        jls.lang.Exception.wrap(e).printStackTrace(jls.lang.System.err);
    }
}

if ('jls.disableDefaultSignalHandler' in _native.core.properties) {
    jls.logger.info('Default signal handlers disabled');
} else {
    /*
     * Register default signal handlers.
     */
    jls.lang.Signal.handle(jls.lang.Signal.SIGINT, function() {
        jls.lang.Runtime.exit(0);
    });
    /*
     * Don't know how to dump javascript thread stacktraces
     */
    /*jls.lang.Signal.handle(jls.lang.Signal.SIGBREAK, function() {
        //jls.lang.System.out.println('... Dump stack trace ...');
        //new jls.lang.Exception().printStackTrace(jls.lang.System.out);
        var name = 'jlsdump.' + new Date().getTime() + '.log';
        jls.lang.System.out.println('... Dump head to ' + name + '...');
        _native.core.dumpHead(name); // Not available in release build
    });*/
}

if ('jls.gcDelay' in _native.core.properties) {
    _native.core._gcThread = new jls.lang.Thread(true); // Daemon
    _native.core._gcThread.run = function() {
        var gcDelay = parseInt(_native.core.properties['jls.gcDelay'], 10);
        if (gcDelay <= 0) {
            jls.logger.warn('Invalid garbage collection delay (' + gcDelay + ')');
            return;
        }
        if (gcDelay < 5000) {
            jls.logger.warn('Aggressive garbage collection delay (' + gcDelay + ')');
        }
        for (;;) {
            jls.lang.Thread.sleep(gcDelay);
            jls.logger.info('start gc...');
            jls.lang.System.gc();
            jls.logger.info('...gc done');
        }
    };
    _native.core._gcThread.start();
} else {
    jls.logger.info('Garbage collection disabled');
}

if ('jls.enableCommonJSModule' in _native.core.properties) {
    jls.loader.require('jls.lang.ModuleLoader');
    require = jls.lang.ModuleLoader.getInstance().require.bind(jls.lang.ModuleLoader.getInstance());
}

if (_native.core.arguments.length > 0) {
    var args = Array.from(_native.core.arguments);
    var arg = args.shift();
    
    if ((arg == null) || (arg == '-help')) {
        _native.core.boot = function() {
            var lines = [
                'Usage: jls [options] classname [args...]',
                'where options include:',
                '\t-bs <bootstrap script file name>',
                '\t\tThe file name of the script to use for bootstrap.',
                '\t-ep <extension path>',
                '\t\tA directory to search for native library files,',
                '\t\tscript ZIP files and directories.',
                '\t-lp <library search path of directories>',
                '\t\tA ; separated list of directories to search for native library files.',
                '\t-sp <script search path of directories and ZIP files>',
                '\t\tA ; separated list of directories and ZIP archives',
                '\t\tto search for script files.',
                '\t-D<name>=<value>',
                '\t\tSet a system property.',
                '\t-logLevel <level>',
                '\t\tSet the native log level.',
                '\t--',
                '\t\tStop the jls options parsing.',
                'where system properties include:',
                '\t-Djls.logger.logLevel=<level>',
                '\t\tSet the log level.',
                '\t-Djls.gcDelay=<delay in ms>',
                '\t\tSet the garbage collection delay, default is disabled.',
                '\t-Djls.disableDefaultExceptionHandler',
                '\t\tDisable uncaught exception default handler.',
                '\t-Djls.disableDefaultSignalHandler',
                '\t\tDisable default signal handlers.',
                '\t-Djls.keep',
                '\t\tWaits Ctrl-C if an exception occurs at boot phase.',
                '\t-Djls.interactive',
                '\t\tEnables interactive boot mode.',
                'See http://javalikescript.free.fr/ for more details.'
            ];
            for (var i = 0; i < lines.length; i++) {
                jls.lang.System.out.println(lines[i]);
            }
        }
    } else if (! arg.endsWith('.js')) {
        /*
         * Override boot function
         */
        _native.core.boot = function() {
            jls.loader.require(arg);
            var mainclass = eval(arg);
            if (! ('main' in mainclass)) {
                throw new jls.lang.Exception('The class ' + arg + ' does not have a main method.');
            }
            mainclass['main'](args);
        };
    }
} else {
    if ('jls.interactive' in _native.core.properties) {
        // TODO
    }
}

if ('jls.keep' in _native.core.properties) {
    var previousBoot = _native.core.boot;
    _native.core.boot = function() {
        try {
            previousBoot.apply(this, arguments);
        } catch (e) {
            jls.lang.System.err.println('Error during boot, hit ctrl-c to end.');
            jls.lang.Exception.wrap(e).printStackTrace(jls.lang.System.err);
            // Start an infinite non daemon thread
            var thread = new jls.lang.Thread();
            thread.run = function() {
                while (true) {
                    jls.lang.Thread.sleep(500);
                }
            };
            thread.start();
        }
    };
}
