jls.loader.provide('jls.gui.Element');

/**
 * @namespace Provides graphical user interface.
 * @see jls.gui.Element
 * @name jls.gui
 */

jls.gui._nsPrefix = '';
jls.gui._classMap = {};
jls.gui.require = function(classname) {
	if (classname in jls.gui._classMap) {
		classname = jls.gui._classMap[classname];
	}
	return jls.loader.require(jls.gui._nsPrefix + classname);
};
switch (_native.core.properties['gui.toolkit']) {
case 'win32':
	jls.gui._nsPrefix = 'jls.win32.';
	jls.gui._classMap = {
			'Button' : 'ButtonElement',
			'CheckBox' : 'CheckBox',
			'ComboBox' : 'ComboBoxElement',
			'Edit' : 'EditElement',
			'Image' : 'ImageElement',
			'Label' : 'Label',
			'MenuItem' : 'MenuItem',
			'Panel' : 'Panel',
			'Tab' : 'TabElement',
			'TextArea' : 'TextAreaElement',
			'TrayIcon' : 'ShellNotifyIconElement',
			'Tree' : 'TreeElement'
	};
	break;
case 'html':
	jls.gui._nsPrefix = 'jls.html.';
	break;
default:
	jls.gui.require = jls.lang.Class.notAvailableFunction;
	break;
}

jls.loader.require('jls.gui.Style');

jls.gui.Element = jls.lang.Class.create( /** @lends jls.gui.Element.prototype */
{
    /**
     * Creates an element.
     * 
     * @param {Object} [parameters] The element parameter.
     * @param {Object} [parameters.attributes] The element attributes.
     * @param {Object} [parameters.style] The element style.
     * @param {Array} [parameters.children] The element children.
     * @param {jls.gui.Element} [parent] The parent.
     * @constructs
	 * @class This class represents a graphical element.
     */
    initialize : function(parameters, parent) {
        parameters = parameters || {};
        this._children = [];
        this._parent = parent || null;
        this._eventsHandlers = {};
        // Default parameters
        this._title = undefined;
        this._id = undefined;
        // Create style
        this._style = new jls.gui.Style(parameters.style);
        // Register style handler
    	this.getStyle().setHandler(this.onStyleChange.bind(this));
        if (this._parent) {
        	this._style.setParent(this._parent.getStyle());
        }
        // Create default layout
    	this._layout = this.createDefaultLayout();
    	// Set bounds
        this._x = this._style.getPropertyValue('left');
        this._y = this._style.getPropertyValue('top');
        this._w = this._style.getPropertyValue('width');
        this._h = this._style.getPropertyValue('height');
        // Set the attributes
        // TODO is this the right place?
        if (parameters.attributes) {
            this.setAttributes(parameters.attributes);
        }
        // Add this element to its parent
        // TODO is this the right place?
        if (this._parent != null) {
            this._parent.addChild(this);
        }
        // Create the concrete element
        this.onCreate();
        // Activate the style properties
        this._style.activate();
        // Add the children
        if (Object.isArray(parameters.children)) {
            this.addChildren(parameters.children);
        }
        if (this._layout != null) {
        	this._layout.onCreate();
        }
    },
    onCreate : jls.lang.Class.emptyFunction,
    getParent : function() {
        return this._parent;
    },
    /*
     * Attributes
     */
    /**
     * Sets an attribute.
     *
     * @param {String} key The attribute key.
     * @param {String} value The attribute value.
     */
    setAttribute : function(key, value) {
        /* look for a setter method in this */
        var setter = key.camelize('set');
        jls.logger.trace('setter: "' + setter + '"');
        if (Object.isFunction(this[setter])) {
            this[setter](value);
            return this;
        }
        this._setAttribute(key, value);
        return this;
    },
    _setAttribute : jls.lang.Class.emptyFunction,
    /**
     * Returns an attribute.
     *
     * @param {String} key The attribute key.
     * @returns {String} The attribute value.
     */
    getAttribute : function(key) {
        jls.logger.trace('getAttribute(' + key + ')');
        /* look for a getter method in this */
        var getter = key.camelize('get');
        jls.logger.trace('getter: "' + getter + '"');
        if (Object.isFunction(this[getter])) {
            return this[getter]();
        }
        return this._getAttribute(key);
    },
    _getAttribute : jls.lang.Class.emptyFunction,
    /**
     * Sets some attributes.
     *
     * @param {Object} attributes The attributes to set.
     */
    setAttributes : function(attributes) {
        for ( var k in attributes) {
            this.setAttribute(k, attributes[k]);
        }
        return this;
    },
    /**
     * Returns the element style.
     * 
     * @returns {jls.gui.Style} The element style.
     */
    getStyle : function() {
        return this._style;
    },
    // onStyleChange(key, oldValue, newValue)
    onStyleChange : jls.lang.Class.emptyFunction,
    /**
     * Returns the layout associated with this element.
     * 
     * @returns {jls.gui.Layout} The element layout or null if there is no layout.
     */
    getLayout : function() {
        return this._layout;
    },
    createDefaultLayout : function() {
        return null;
    },
    /**
     * Sets the layout for this element.
     * 
     * @param {jls.gui.Layout} layout The layout to associate to this element.
     */
    setLayout : function(layout) {
    	if (layout != null) {
    		if (typeof layout == 'string') {
    			var layoutClass = jls.loader.require(layout);
    			layout = new layoutClass(this);
    		} else if (layout instanceof jls.gui.Layout) {
    			layout.setElement(this);
    		} else {
    			throw new jls.lang.Exception('Invalid layout');
    		}
    	}
        this._layout = layout;
        return this;
    },
    /*
     * Element hierarchy
     */
    /**
     * Adds a child.
     *
     * @param {Object} child The child to add.
     */
    addChild : function(child) {
        if (child instanceof jls.gui.Element) {
            jls.logger.trace('addChild for jls.gui.Element');
            // TODO if the child already have a parent remove it
            child._parent = this;
            this._children.push(child);
        } else if ((typeof child == "object") && ('classname' in child)) {
            jls.logger.trace('addChild for object');
            child = jls.lang.Class.newInstance(child, this);
        } else {
            throw new jls.lang.Exception('don\'t know how to add "' + (typeof child) + '" child type');
        }
        if (this._layout != null) {
        	this._layout.onAddChild(child);
        }
        return child;
    },
    /**
     * Adds children.
     *
     * @param {Object} children The children to add.
     */
    addChildren : function(children) {
        for ( var i = 0; i < children.length; i++) {
            this.addChild(children[i]);
        }
        return this;
    },
    /**
     * Removes a child.
     *
     * @param {Object} child The child to remove.
     */
    removeChild : function(child) {
        var index = 0;
        if (typeof child == 'number') {
            index = child;
        } else {
            for (index = this._children.length - 1; (index >= 0) && (this._children[index] !== child); index--);
            if (index < 0) {
                throw new jls.lang.Exception('Child not found');
            }
        }
        var removed = this._children[index];
        this._children.splice(index, 1);
        //jls.logger.warn('removeChild(' + index + '))');
        if (removed && (removed instanceof jls.gui.Element)) {
            removed._parent = null;
            removed.onDestroy();
        }
        if (this._layout != null) {
        	this._layout.onRemoveChild(child);
        }
        return removed;
    },
    /**
     * Removes all children.
     *
     */
    removeChildren : function() {
        for ( var i = this._children.length - 1; i >= 0; i--) {
            this.removeChild(i);
        }
    },
    /**
     * Returns all children.
     *
     */
    getChildren : function() {
        return this._children;
    },
    getChildCount : function() {
        return this._children.length;
    },
    /**
     * Gets a child.
     *
     * @param {Number} index The child index.
     * @returns {Object} The child.
     */
    getChild : function(index) {
        return this._children[index];
    },
    /**
     * Destroys this element.
     *
     */
    destroy : function() {
        if (this.getParent() != null) {
            this.getParent().removeChild(this);
        } else {
            this.onDestroy();
        }
        return this;
    },
    onDestroy : jls.lang.Class.emptyFunction,
    /*
     * Event.
     * TODO Move this in a specialized event broker class.
     */
    dispatch : function(event) {
        if (event.type in this._eventsHandlers) {
            var handlers = this._eventsHandlers[event.type];
            for (var i = handlers.length - 1; i >= 0; i--) {
                handlers[i](event);
            }
        }
        return this;
    },
    onStartObserving: jls.lang.Class.emptyFunction,
    onStopObserving: jls.lang.Class.emptyFunction,
    /**
     * Register an event handler.
     *
     * @param {String} type The event type to observe.
     * @param {Function} handler The handler to register.
     * @returns {Function} The handler.
     */
    observe: function(type, handler) {
        if (Object.keys(this._eventsHandlers).length == 0) {
            this.onStartObserving();
        }
        if (! (type in this._eventsHandlers)) {
            this._eventsHandlers[type] = [];
        }
        this._eventsHandlers[type].push(handler);
        return handler;
    },
    /**
     * Unregister an event handler.
     *
     * @param {String} type The event type to unregister.
     * @param {Function} handler The handler to unregister.
     * @returns {Array} The removed handlers.
     */
    unobserve: function(type, handler) {
        var removed = [];
        if (type in this._eventsHandlers) {
            var handlers = this._eventsHandlers[type];
            for (var i = handlers.length - 1; i >= 0; i--) {
                if (handler == handlers[i]) {
                    removed.push(handlers[i]);
                    handlers.splice(i, 1);
                }
            }
            if (handlers.length == 0) {
                delete this._eventsHandlers[type];
            }
        }
        if (Object.keys(this._eventsHandlers).length == 0) {
            this.onStopObserving();
        }
        return removed;
    },
    /*
     * Update
     */
    // TODO doLayout ?
    update : function() {
        if (this._layout != null) {
        	this._layout.onUpdate();
        }
    },
    /*
     * Default attributes
     */
    getX : function() {
        return this._x;
    },
    getY : function() {
        return this._y;
    },
    getW : function() {
        return this._w;
    },
    getH : function() {
        return this._h;
    },
    getBounds : function() {
        return [this._x, this._y, this._w, this._h];
    },
    setBounds : function(rect) {
        this._x = rect[0];
        this._y = rect[1];
        this._w = rect[2];
        this._h = rect[3];
        return this;
    },
    getLocation : function() {
        return [this._x, this._y];
    },
    setLocation : function(point) {
        this._x = point[0];
        this._y = point[1];
        return this;
    },
    getSize : function() {
        return [this._w, this._h];
    },
    setSize : function(dim) {
        this._w = dim[0];
        this._h = dim[1];
        return this;
    },
    getClientOffset : function() {
        return [0, 0];
    },
    /*getClientBounds : function() {
        return this.getBounds();
    },*/
    getClientSize : function() {
        return this.getSize();
    },
    getTitle : function() {
        return this._title;
    },
    setTitle : function(title) {
        this._title = title;
        return this;
    }
});
