jls.loader.provide('jls.io.File');

jls.loader.requireLibrary('jls_io');

jls.io.File = jls.lang.Class.create( /** @lends jls.io.File.prototype */
{
    /**
     * Creates a file.
     * 
     * @param {jls.io.File} [parent] The parent file.
     * @param {String} child The file pathname.
     * @constructs
     * @class This class represents a file system entry.
     */
    initialize : function(parent, child) {
        if (! child) {
            child = parent;
            parent = undefined;
        }
        if (typeof child != 'string') {
            throw new jls.lang.Exception('Invalid File argument type (' + (typeof child) + ')');
        }
        if (parent) {
            if (parent instanceof jls.io.File) {
                this.path = jls.io.File.concat(parent.getPath(), jls.io.File.normalize(child));
            } else if (typeof parent == 'string') {
                this.path = jls.io.File.normalize(jls.io.File.concat(parent, child));
            }
        } else {
            this.path = jls.io.File.normalize(child);
        }
    },
    /**
     * Removes the file entry.
     *
     * @returns {Boolean} true if the entry is removed.
     */
    remove : function() {
        return _native.io.delete(this.path);
    },
    /**
     * Tests whether this file exists.
     *
     * @returns {Boolean} true if the file exists.
     */
    exists : function() {
        return _native.io.exists(this.path);
    },
    /**
     * Returns the name of this file. This is the last name in the pathname's name sequence.
     *
     * @returns {String} The name of this file.
     */
    getName : function() {
	    var index = this.path.lastIndexOf(jls.io.File.separator);
	    if (index > 0) {
		    return this.path.substr(index + 1);
	    } else {
		    return this.path;
	    }
    },
    /**
     * Returns the extension of this file.
     *
     * @returns {String} The extension of this file.
     */
    getExtension : function() {
        var index = this.path.lastIndexOf('.');
	    if (index > 0) {
		    return this.path.substr(index + 1);
	    } else {
		    return '';
	    }
    },
    /**
     * Returns the path of this file.
     *
     * @returns {String} The path of this file.
     */
    getPath : function() {
        return this.path;
    },
    getParent : function() {
        return this.getParentFile().getPath();
    },
    getParentFile : function() {
    	if (this.isAbsolute()) {
    		if (jls.io.File.isRoot(this.path)) {
    			return null;
    		}
    		return new jls.io.File(this.path.substring(0, this.path.lastIndexOf(jls.io.File.separator)));
    	}
        throw new jls.lang.Exception('Not supported');
    },
    /**
     * Tests whether the file path is absolute.
     *
     * @returns {Boolean} true if the file path is absolute.
     */
    isAbsolute : function() {
        return this.path.startsWith(jls.io.File.separator) || (this.path.indexOf(':\\') == 1);
    },
    /**
     * Tests whether the file entry is a directory.
     *
     * @returns {Boolean} true if the file entry is a directory.
     */
    isDirectory : function() {
        return _native.io.getType(this.path) == _native.io.TYPE_DIRECTORY;
    },
    /**
     * Tests whether the file entry is a file.
     *
     * @returns {Boolean} true if the file entry is a file.
     */
    isFile : function() {
        return _native.io.getType(this.path) == _native.io.TYPE_FILE;
    },
    isHidden : function() {
        throw new jls.lang.Exception('Not supported');
        return false;
    },
    lastModified : function() {
        return _native.io.getModifyTime(this.path);
    },
    /**
     * Returns the length of the file entry represented by this file.
     *
     * @returns {Number} The length of the file entry represented by this file.
     */
    length : function() {
        return _native.io.getSize(this.path);
    },
    /**
     * Returns an array of strings naming the file system entries in the directory represented by this file.
     *
     * @returns {Array} An array of strings naming the file system entries.
     */
    list : function() {
        return _native.io.list(this.path);
    },
    /**
     * Returns an array of files in the directory represented by this file.
     *
     * @returns {Array} An array of files.
     */
    listFiles : function() {
        var l = this.list();
        var lf = [];
        for (var i = 0; i < l.length; i++) {
            lf.push(new jls.io.File(this, l[i]));
        }
        return lf;
    },
    getAbsoluteFile : function() {
        throw new jls.lang.Exception('Not supported');
        return false;
    },
    /**
     * Returns the absolute path of this file.
     *
     * @returns {String} The absolute path of this file.
     */
    getAbsolutePath : function() {
	    if (this.isAbsolute()) {
		    return this.getPath();
	    } else {
    	    return jls.io.File.concat(jls.io.File.userDir, this.getPath());
	    }
    },
    /**
     * Creates the directory named by this file.
     *
     * @returns {Boolean} true if the directory is created.
     */
    mkdir : function() {
        return _native.io.mkDir(this.path);
    },
    mkdirs : function() {
        throw new jls.lang.Exception('Not supported');
        return false;
    },
    /**
     * Renames this file.
     *
     * @param {String} newPath The new path for this file.
     * @returns {Boolean} true if this file is renamed.
     */
    renameTo : function(newPath) {
        return _native.io.rename(this.path, newPath);
    },
    setLastModified : function() {
        throw new jls.lang.Exception('Not supported');
        return false;
    }
});


Object.extend(jls.io.File, /** @lends jls.io.File */
{
    isRoot : function(path) {
        return (path != null) && ((path == '/') || ((path.length == 3) && path.endsWith(':\\')));
    },
    /**
     * Concats to path parts and then returns the resulting path.
     *
     * @param {String} a The first path part to concat.
     * @param {String} b The last path part to concat.
     * @returns {String} The concated path.
     */
    concat : function(a, b) {
        return a + jls.io.File.separator + b;
    },
    /**
     * Normalizes a path.
     *
     * @param {String} path The path to normalize.
     * @returns {String} The normalized path.
     */
    normalize : function(path) {
        var npath = path.replace(/[\\\/][\\\/]*/g, jls.io.File.separator);
        if ((! jls.io.File.isRoot(path)) && (npath.length > 0) && (npath.endsWith(jls.io.File.separator))) {
            npath = npath.substr(0, npath.length - 1 - jls.io.File.separator.length);
        }
        return npath;
    },
    createTempFile : function() {
        throw new jls.lang.Exception('Not supported');
        return false;
    },
    /**
     * Returns an array of file system roots.
     *
     * @returns {Array} An array of files denoting the available filesystem roots.
     */
    listRoots : function() {
        var list = [];
        var roots = _native.io.listRoots();
        for (var i = 0; i < roots.length; i++) {
            list.push(new jls.io.File(roots[i]));
        }
        return list;
    },
    userDir : _native.core.properties['user.dir'],
    pathSeparator : _native.core.properties['path.separator'],
    separator : _native.core.properties['file.separator']
});

