jls.loader.provide('jls.io.FileDescriptor');

jls.loader.requireLibrary('jls_io');

/**
 * @class This class represents a file system descriptor.
 * A file descriptor represents I/O objects, such as open files and sockets. 
 */
jls.io.FileDescriptor = jls.lang.Class.create(/** @lends jls.io.FileDescriptor.prototype */
{
    initialize : function(nfd) {
        if (! ((nfd instanceof _native.io.FileDesc) || (nfd instanceof _native.io.StaticFileDesc))) {
            throw new jls.lang.Exception('Invalid file descriptor type');
        }
        this._fd = nfd;
    },
    setInheritable : function(inheritable) {
        this._fd.setInheritable(inheritable);
        return this;
    },
    /**
     * Closes this file.
     *
     */
    close : function() {
        this._fd.close();
        return this;
    },
    /**
     * Flushs this file.
     *
     */
    flush : function() {
        return this;
    },
    /**
     * Tells if this stream supports the mark and reset methods.
     * 
     * @returns {Boolean} if this stream instance supports the mark and reset methods; false otherwise.
     */
    markSupported : function() {
        return ('mark' in this) && ('reset' in this);
    },
    /*
     * Seek.
     * 
     * @param {Number} offset ?.
     * @param {Number} [whence] ?.
     * @returns {Number} The position.
     */
    seek : function(offset, whence) {
        if (typeof whence == 'undefined') {
            whence = jls.io.FileDescriptor.SEEK_SET;
        }
        return this._fd.seek(offset, whence);
    },
    skipBytes : function(n) {
        return 0; // TODO ?
    },
    /**
     * Reads a byte.
     * 
     * @returns {Number} The unsigned byte or -1.
     */
    readByte : function() {
        return this._fd.readByte();
    },
    /**
     * Writes a byte.
     * 
     * @param {Number} b The byte to write.
     * @returns {Boolean} true if the byte has been write.
     */
    writeByte : function(b) {
        return this._fd.writeByte(b);
    },
    /**
     * Reads bytes into the specified byte array, starting at the given offset.
     * 
     * @param {ByteArray} barray The destination byte array.
     * @param {Number} offset The offset at which to start storing bytes.
     * @param {Number} length The maximum number of bytes to read.
     * @returns {Number} The total number of bytes read.
     */
    readByteArray : function(barray, offset, length) {
    	offset = offset || 0;
    	length = length || barray.size() - offset;
        return this._fd.read(barray, offset, length);
    },
    /**
     * Writes bytes from the specified byte array starting at the given offset.
     * 
     * @param {ByteArray} barray The source byte array.
     * @param {Number} offset The offset at which to start getting bytes.
     * @param {Number} length The maximum number of bytes to write.
     * @returns {Number} The number of bytes written.
     */
    writeByteArray : function(barray, offset, length) {
    	offset = offset || 0;
    	length = length || barray.size() - offset;
        return this._fd.write(barray, offset, length);
    },
    /**
     * Reads this file into a buffer.
     *
     * @param {jls.lang.Buffer} buffer The buffer to read.
     * @returns {Number} the read byte count.
     */
    read : function(buffer) {
        var count = this._fd.read(buffer.byteArray(), buffer.offset(), buffer.remaining());
        if (count > 0) {
            buffer.incrementPosition(count);
        }
        return count;
    },
    /**
     * Writes a buffer into this file.
     *
     * @param {jls.lang.Buffer} buffer The buffer to write.
     * @returns {Number} the write byte count.
     */
    write : function(buffer) {
        var count = this._fd.write(buffer.byteArray(), buffer.offset(), buffer.remaining());
        if (count > 0) {
            buffer.incrementPosition(count);
        }
        return count;
    },
    // TODO Remove
    writeString : function(s) {
        return this._fd.writeString(s);
    },
    getFD : function() {
        return this._fd;
    }
});

Object.extend(jls.io.FileDescriptor, /** @lends jls.io.FileDescriptor */
{
    SEEK_SET : _native.io.FileDesc.SEEK_SET,
    SEEK_CUR : _native.io.FileDesc.SEEK_CUR,
    SEEK_END : _native.io.FileDesc.SEEK_END
});
