//jls.loader.provide('jls.lang.Exception');

jls.lang.Exception = jls.lang.Class.create(/** @lends jls.lang.Exception.prototype */
{
    /**
     * @param {String} [message] The detail message.
     * @param {Object} [cause] The cause.
     * @constructs
     * @class Provide the base Exception.
     */
    initialize : function(message, cause, name) {
        /**
         * The detail message
         * 
         * @private
         * @type String
         */
        this._message = message || null;
        /**
         * The cause
         * 
         * @private
         * @type Object
         */
        this._cause = null;
        if (cause) {
            this._cause = jls.lang.Exception.wrap(cause);
        }
        /*
         * Generate a JS Error to get the stacktrace.
         * The error fields are: name, message, fileName, lineNumber, stack
         * Then remove extraenous stack information.
         */
        this._stack = jls.lang.Exception.getStackFromError(new Error());
        this._stack.shift(); // Remove Error stack
        this._name = name || null;
    },
    /**
     * Returns the cause of this exception or null if the cause is nonexistent or unknown. The cause is the exception that caused this
     * exception to get thrown.
     * 
     * @returns {Object} The cause.
     */
    getCause : function() {
        return this._cause;
    },
    /**
     * Returns the detail message string of this exception.
     * 
     * @returns {String} The detail message.
     */
    getMessage : function() {
        return this._message;
    },
    getName : function() {
        return this._name || jls.loader.getClassname(this) || 'jls.lang.Exception';
    },
    /**
     * Returns the stack trace of this exception.
     * 
     * @returns {Array} The stack trace.
     */
    getStackTrace : function() {
        return this._stack;
    },
    /**
     * Prints this exception and its stacktrace.
     * 
     * @param {jls.io.PrintStream} [ps] The print stream to use.
     */
    printStackTrace : function(ps) {
        ps = ps || jls.lang.System.out;
        var last = null;
        for (var e = this; e != null; e = e.getCause()) {
            if (e == this) {
                ps.println(e.toString());
            } else {
                ps.println('Caused by: ' + e.toString());
            }
            var st = e.getStackTrace();
            var steStr = null;
            for (var i = 0; i < st.length; i++) {
                var ste = st[i]
                steStr = '\tat ' + ste.fileName + ':' + ste.lineNumber;
                if (ste.arguments && (ste.arguments.length > 2)) {
                    steStr += ' called with ' + ste.arguments;
                }
                if (steStr == last) {
                    ps.println('\t... ' + (st.length - i) + ' more');
                    break;
                }
                ps.println(steStr);
            }
            last = steStr;
        }
    },
    /**
     * Returns a short description of this exception.
     * 
     * @returns {String} The detail message.
     */
    toString : function() {
        var msg = this.getMessage();
        if (msg == null) {
            return this.getName();
        } else {
            return this.getName() + ': ' + msg;
        }
    }
});

Object.extend(jls.lang.Exception, /** @lends jls.lang.Exception */
{
    wrap : function(e) {
        if (e instanceof jls.lang.Exception) {
            return e;
        } else if ((e instanceof Error) || (e instanceof TypeError) || (e instanceof SyntaxError)) {
            var ex = new jls.lang.Exception(e.message, undefined, e.name);
            ex._stack = jls.lang.Exception.getStackFromError(e);
            return ex;
        } else {
            return new jls.lang.Exception(e == null ? undefined : e.toString());
        }
    },
    getStackFromError : function(e) {
    	/*for (var k in e) {
    		jls.logger.debug(k + ': ' + e[k].toString());
    	}*/
        var eStack = e.stack.split('\n');
        var stack = [];
        var i = 0;
        if (('fileName' in e) && ('lineNumber' in e) && (eStack[i] == '@:0')) {
        	// The error comes from a native call evaluation and includes filename and line number
        	i++;
            stack.push({
                arguments: '',
                fileName: e.fileName,
                lineNumber: e.lineNumber
            });
        }
        for (; i < eStack.length; i++) {
            if (eStack[i].length == 0) {
                continue;
            }
            var ste = eStack[i].split('@');
            if (ste[1] == ':0') {
                continue;
            }
            var fan = ste[1].split(':');
            if (fan[0] == 'bootstrap.js') {
                continue;
            }
            stack.push({
                arguments: ste[0],
                fileName: fan[0],
                lineNumber: fan[1]
            });
        }
        return stack;
    }
});

