jls.loader.provide('jls.lang.Process');

jls.lang.Process = jls.lang.Class.create( /** @lends jls.lang.Process.prototype */
{
    /**
     * Creates a process with the specified command and arguments with the specified environment and working directory.
     * 
     * @param {Array} cmdArray Array of strings specifying the command-line arguments. The first argument is the name of the executable file.
     * @param {Array} envp Array of key-values specifying the environment strings. If undefined, the new process inherits the environment of the parent process.
     * @param {jls.io.File} dir The working directory of the subprocess, or undefined if the subprocess should inherit the working directory of the current process.
     * @constructs
     * @class The class Process provides methods for performing input from the process,
     * performing output to the process, waiting for the process to complete,
     * checking the exit status of the process, and destroying (killing) the process.
     */
    initialize : function(cmdArray, envp, processAttr) {
        var cmd = (typeof cmdArray == 'string') ? [cmdArray] : cmdArray;
        this._no = new _native.core.Process(cmd[0], cmd, envp, processAttr);
        this._exitValue = jls.lang.Process.STILL_ACTIVE;
        this._thread = null;
        this._onExitCBs = [];
    },
    /**
     * Kills this process.
     */
    destroy : function() {
    	if (this._no == null) {
            throw new jls.lang.Exception('The process is no more reachable');
    	}
        this._no.kill();
        this._exitValue = jls.lang.Process.DESTROYED;
        this._no = null;
    },
    /**
     * Detachs this process.
     */
    detach : function() {
    	if (this._no == null) {
            throw new jls.lang.Exception('The process is no more reachable');
    	}
        this._no.detach();
        this._exitValue = jls.lang.Process.DETACHED;
        this._no = null;
    },
    /**
     * Waits for this process to terminate and return the exit value.
     * @returns {Number} This process exit value.
     */
    waitFor : function() {
    	if (this._no == null) {
            throw new jls.lang.Exception('The process is no more reachable');
    	}
    	if (this._thread == null) {
            this._exitValue = this._no.wait();
    	} else {
    		this._thread.join();
    	}
        return this._exitValue;
    },
    /**
     * Registers an exit callback.
     * @param {Function} [fn] An exit callback.
     */
    registerExitCallback : function(fn) {
    	if (this._no == null) {
            throw new jls.lang.Exception('This process is no more reachable');
    	}
    	this._onExitCBs.push(fn);
    	if (this._thread == null) {
            this._thread = new jls.lang.Thread();
            this._thread.run = function() {
            	jls.logger.debug('waitFor()...');
            	try {
                	this._exitValue = this._no.wait();
            	}
            	catch (e) {
            		this._exitException = e;
            	}
            	jls.logger.debug('exitCode: ' + this._exitValue);
            	this._thread = null;
            	this._no = null;
            	for (var i = 0; i < this._onExitCBs.length; i++) {
                	if (this._onExitCBs[i]) {
                		this._onExitCBs[i].call(this, this._exitValue);
                	}
            	}
            };
            jls.logger.debug('Start exit callback thread');
            this._thread.start(this);
    	}
        return this;
    },
    /**
     * Unregisters an exit callback.
     * @param {Function} [fn] An exit callback.
     */
    unregisterExitCallback : function(fn) {
        for (var i = this._onExitCBs.length - 1; i >= 0; i--) {
            if (fn === this._onExitCBs[i]) {
                this._onExitCBs.splice(i, 1);
                break;
            }
        }
        return this;
    },
    /**
     * Returns this process exit value.
     * @returns {Number} This process exit value.
     */
    exitValue : function() {
    	if (this._exitValue == jls.lang.Process.STILL_ACTIVE) {
            throw new jls.lang.Exception('The process is still active');
    	}
        return this._exitValue;
    }
});

Object.extend(jls.lang.Process,
{
	DESTROYED: 256,
	DETACHED: 257,
	STILL_ACTIVE: 259
});


