jls.loader.provide('jls.lang.ProcessBuilder');

jls.loader.require('jls.lang.Process');

jls.lang.ProcessBuilder = jls.lang.Class.create( /** @lends jls.lang.ProcessBuilder.prototype */
{
    /**
     * Creates a process builder with the specified command and arguments with the specified environment and working directory.
     * 
     * @param {Array} cmdArray Array of strings specifying the command-line arguments. The first argument is the name of the executable file.
     * @param {Array} [envp] Array of key-values specifying the environment strings. If undefined, the new process inherits the environment of the parent process.
     * @constructs
     * @class This class is used to create OS processes.
     */
    initialize : function(cmdArray, envp) {
        this._no = undefined;
        this._cmdArray = cmdArray;
        this._envp = envp;
        this._process = null;
        this._stdoutHandler = null;
    },
    /**
     * Sets this process builder's working directory.
     * 
     * @param {jls.io.File} dir The working directory of the subprocess, or undefined if the subprocess should inherit the working directory of the current process.
     */
    setCurrentDirectory : function(dir) {
        if (! this._no) {
            this._no = new _native.core.ProcessAttr();
        }
        this._no.setCurrentDirectory(dir.getAbsolutePath());
    },
    /**
     * Sets this process builder's working directory.
     * 
     * @param {jls.io.FileDescriptor} fd The file descriptor to use for redirection.
     * @param {Number} stdio The standard IO to redirect.
     */
    setStdioRedirect : function(fd, stdio) {
        if (! this._no) {
            this._no = new _native.core.ProcessAttr();
        }
        fd.setInheritable(true);
        this._no.setStdioRedirect(fd.getFD(), stdio || jls.io.Pipe.StandardOutput);
    },
    setReadHandler : function(fn, stdio, bufferSize) {
    	bufferSize = bufferSize || 1024;
    	var handler = this._stdoutHandler = {};
    	handler.ended = false;
    	handler.pipe = new jls.io.Pipe();
    	handler.thread = new jls.lang.Thread();
    	this.setStdioRedirect(handler.pipe.source(), jls.lang.ProcessBuilder.StandardOutput);
    	handler.thread.run = function() {
        	var buffer = jls.lang.ByteBuffer.allocate(bufferSize);
        	for (;;) {
        		buffer.clear();
            	this.pipe.sink().read(buffer);
            	buffer.flip();
            	if (this.ended) {
            		if (buffer.remaining() > 1) {
                        jls.logger.warn('unreaded data in the pipe: ' + (buffer.remaining() - 1));
            		}
            		break;
            	}
            	while (buffer.remaining() > 0) {
                	fn(buffer);
            	}
        	}
        };
    },
    /**
     * Starts a new process using the attributes of this process builder.
     * 
     * @returns {jls.lang.Process} A new Process for managing the subprocess.
     */
    start : function() {
    	this._process = new jls.lang.Process(this._cmdArray, this._envp, this._no);
    	this._process._processBuilder = this;
    	if (this._stdoutHandler != null) {
            jls.logger.debug('Start reading thread');
            this._stdoutHandler.thread.start(this._stdoutHandler);
            this._process.registerExitCallback(function(exitValue) {
                jls.logger.debug('Stop reading thread (' + exitValue + ')');
            	// say that its over
            	this._processBuilder._stdoutHandler.ended = true;
            	// unlock pipe reader
            	this._processBuilder._stdoutHandler.pipe.source().writeByte(0);
            });
    	}
        return this._process;
    }
});

Object.extend(jls.lang.ProcessBuilder, /** @lends jls.lang.ProcessBuilder */
{
    /**
     * Represents the standard output.
     * 
     * @type Number
     */
    StandardOutput : _native.core.ProcessAttr.StandardOutput,
    /**
     * Represents the standard error.
     * 
     * @type Number
     */
    StandardError : _native.core.ProcessAttr.StandardError,
    /**
     * Represents the standard input.
     * 
     * @type Number
     */
    StandardInput : _native.core.ProcessAttr.StandardInput
});

