jls.loader.provide('jls.lang.Runtime');

jls.loader.require('jls.lang.Process');
jls.loader.require('jls.lang.ProcessBuilder');

/**
 * @class The Runtime class provides interaction with the underlying OS.
 */
jls.lang.Runtime = jls.lang.Class.create({});

Object.extend(jls.lang.Runtime, /** @lends jls.lang.Runtime */
{
    /**
     * Executes the specified command and arguments in a separate process with the specified environment and working directory.
     * 
     * @param {Array} cmdArray Array of strings specifying the command-line arguments. The first argument is the name of the executable file.
     * @param {Array} envp Array of key=values specifying the environment strings. If undefined, the new process inherits the environment of the parent process.
     * @param {jls.io.File} dir The working directory of the subprocess, or undefined if the subprocess should inherit the working directory of the current process.
     * @returns {jls.lang.Process} A new Process for managing the subprocess.
     */
    exec : function(cmdArray, envp, dir) {
        var pb = new jls.lang.ProcessBuilder(cmdArray, envp);
        if (dir) {
            pb.setCurrentDirectory(dir);
        }
        return pb.start();
    },
    /**
     * Terminates the program and returns a value to the OS.
     * 
     * @param {Number} status The exit status to return to the OS.
     */
    exit : function(status) {
        _native.core.exit(status);
    },
    /**
     * Forcibly terminates the program and returns a value to the OS.
     * 
     * @param {Number} status The exit status to return to the OS.
     */
    halt : function(status) {
        _native.core.halt(status);
    },
    /**
     * Runs the garbage collector.
     */
    gc : function() {
        _native.core.gc();
    },
    /**
     * Loads a native library compliant with the JLS native engine.
     * 
     * @param {String} name The name of the library to load.
     */
    loadLibrary : function(libname) {
        jls.loader.requireLibrary(libname);
    },
    _shutdownHooks : [],
    _shutdownHook : function() {
        for (var i = 0; i < jls.lang.Runtime._shutdownHooks.length; i++) {
            try {
                jls.lang.Runtime._shutdownHooks[i]();
            } catch (e) {
                jls.logger.warn('uncaught exception while calling shutdown hook \"' + e + '\"');
            }
        }
    },
    /**
     * Registers a new virtual-machine shutdown hook.
     * 
     * @param {Function} hook The hook function.
     */
    addShutdownHook : function(hook) {
        if (! ('shutdownHook' in _native.core)) {
            _native.core.shutdownHook = jls.lang.Runtime._shutdownHook;
        }
        jls.lang.Runtime._shutdownHooks.push(hook);
    },
    /**
     * Gets the current runtime instance.
     * 
     * @returns {jls.lang.Runtime} The runtime instance.
     */
    getRuntime : function() {
        return jls.lang.Runtime;
    }
});

