jls.loader.provide('jls.lang.System');

jls.loader.requireLibrary('jls_io');
jls.loader.require('jls.lang.Runtime');
jls.loader.require('jls.io.PrintStream');
jls.loader.require('jls.io.FileInputStream');

/**
 * @class The System class provides access to data and operations of the underlying OS.
 * Access to system properties, environnement variables and default standard IO streams.
 * Operation such as the exit method, the garbage collection and the ability to load native library.
 */
jls.lang.System = jls.lang.Class.create({});

Object.extend(jls.lang.System, /** @lends jls.lang.System */
{
    _systemProperties : null,
    _commandLineArguments : null,
    _mainClassname : null,
    /**
     * Initialize the system class.
     * @private
     */
    _initialize : function() {
        jls.lang.System.err = new jls.io.PrintStream(new jls.io.FileOutputStream(_native.io.stderr));
        jls.lang.System.in = new jls.io.FileInputStream(_native.io.stdin);
        jls.lang.System.out = new jls.io.PrintStream(new jls.io.FileOutputStream(_native.io.stdout));
        jls.lang.System._systemProperties = _native.core.properties;
        jls.lang.System._commandLineArguments = Array.from(_native.core.arguments);
        jls.lang.System._mainClassname = jls.lang.System._commandLineArguments.shift();
    },
    /**
     * Gets the system command-line arguments.
     * 
     * @returns {Array} The system arguments.
     */
    getArguments : function() {
        return jls.lang.System._commandLineArguments;
    },
    /**
     * Gets the system properties.
     * 
     * @returns {Object} The system properties.
     */
    getProperties : function() {
        return jls.lang.System._systemProperties;
    },
    /**
     * Gets a specific system property.
     * 
     * @param {String} name The name of the property to get.
     * @returns {String} The system property.
     */
    getProperty : function(name, def) {
    	if (name in jls.lang.System._systemProperties) {
            return jls.lang.System._systemProperties[name];
    	}
    	return def;
    },
    /**
     * Removes a specific system property.
     * 
     * @param {String} name The name of the property to remove.
     * @returns {String} The system property removed.
     */
    clearProperty : function(name) {
        var cleared = null;
        if (name in jls.lang.System._systemProperties) {
            cleared = jls.lang.System._systemProperties[name];
            delete jls.lang.System._systemProperties[name];
        }
        return cleared;
    },
    /**
     * Sets a specific system property.
     * 
     * @param {String} name The name of the property to set.
     * @param {String} value The value of the property to set.
     */
    setProperty : function(name, value) {
        jls.lang.System._systemProperties[name] = value;
    },
    /**
     * Gets a specific environnement property.
     * 
     * @param {String} name The name of the property to get.
     * @returns {String} The environnement property.
     */
    getenv : function(name) {
        return _native.core.getEnv(name);
    },
    /**
     * Returns the current time in milliseconds.
     * 
     * @returns {Number} The current time in milliseconds.
     */
    currentTimeMillis : function() {
        return new Date().getTime();
    },
    /**
     * The standard error stream.
     * @type jls.io.PrintStream
     */
    err : null,
    /**
     * The standard input stream.
     * @type jls.io.FileInputStream
     * @fieldOf jls.lang.System
     * @name in
     */
    'in' : null,
    /**
     * The standard output stream.
     * @type jls.io.PrintStream
     */
    out : null,

    /**
     * Terminates the program and returns a value to the OS.
     * 
     * @param {Number} status The exit status to return to the OS.
     * @function
     */
    exit : jls.lang.Runtime.exit,
    /**
     * Runs the garbage collector.
     * @function
     */
    gc : jls.lang.Runtime.gc,
    /**
     * Loads a native library compliant with the JLS native engine.
     * 
     * @param {String} name The name of the library to load.
     * @function
     */
    loadLibrary : jls.lang.Runtime.loadLibrary
});

