jls.loader.provide('jls.net.Socket');

jls.loader.requireLibrary('jls_net');
jls.loader.require('jls.net.InetAddress');
jls.loader.require('jls.net.InetSocketAddress');

jls.net.Socket = jls.lang.Class.create( /** @lends jls.net.Socket.prototype */
{
    /**
     * Creates a TCP socket.
     *
     * @param {String} [host] The host name to connect to, could be a jls.net.InetAddress.
     * @param {Number} [port] The port number to connect to.
     * @constructs
	 * @class This class represents a client TCP socket.
	 * A socket is an endpoint for communication between two machines.
	 * TCP (Transmission Control Protocol) is a connection-oriented, reliable byte-stream protocol of the TCP/IP protocol suite.
     */
    initialize : function(host, port) {
        this._fd = null;
        this._address = null;
        if (typeof host == 'undefined') {
            // nothing to do
        } else if (host instanceof _native.net.Socket) {
            this._fd = host;
        } else if (host instanceof jls.net.InetSocketAddress) {
            this.connect(host);
        } else if ((host instanceof jls.net.InetAddress) || (typeof host == 'string')) {
            this.connect(new jls.net.InetSocketAddress(host, port));
        } else {
            throw new jls.lang.Exception('Invalid arguments');
        }
    },
    /**
     * Connects this socket to the server.
     * 
     * @returns {jls.net.InetSocketAddress} The socket address of the server.
     */
    connect : function(endpoint) {
        if (! (endpoint instanceof jls.net.InetSocketAddress)) {
            throw new jls.lang.Exception('Invalid socket address');
        }
        if (this._fd == null) {
            this._fd = new _native.net.Socket();
        }
        this._address = endpoint;
        this._fd.connect(endpoint.getAddress().getHostAddress(), endpoint.getPort());
        return this;
    },
    getRemoteSocketAddress : function() {
        return this._address;
    },
    /**
     * Closes this socket.
     * 
     * @returns {jls.net.Socket} This socket.
     */
    close : function() {
        this._fd.close();
        return this;
    },
    /**
     * Flushs this socket.
     * 
     * @returns {jls.net.Socket} This socket.
     */
    flush : function() {
        return this;
    },
    /**
     * Reads a byte.
     * 
     * @returns {Number} The unsigned byte or -1.
     */
    readByte : function() {
        return this._fd.recvByte();
    },
    /**
     * Writes a byte.
     * 
     * @param {Number} b The byte to write.
     * @returns {Boolean} true if the byte has been write.
     */
    writeByte : function(b) {
        return this._fd.sendByte(b);
    },
    /**
     * Reads bytes into the specified byte array, starting at the given offset.
     * 
     * @param {ByteArray} barray The destination byte array.
     * @param {Number} offset The offset at which to start storing bytes.
     * @param {Number} length The maximum number of bytes to read.
     * @returns {Number} The total number of bytes read.
     */
    readByteArray : function(barray, offset, length) {
    	offset = offset || 0;
    	length = length || barray.size() - offset;
        return this._fd.recv(barray, offset, length);
    },
    /**
     * Writes bytes from the specified byte array starting at the given offset.
     * 
     * @param {ByteArray} barray The source byte array.
     * @param {Number} offset The offset at which to start getting bytes.
     * @param {Number} length The maximum number of bytes to write.
     * @returns {Number} The number of bytes written.
     */
    writeByteArray : function(barray, offset, length) {
    	offset = offset || 0;
    	length = length || barray.size() - offset;
        return this._fd.send(barray, offset, length);
    },
    /**
     * Reads a sequence of bytes from this channel into the given buffer.
     * 
     * @param {jls.lang.Buffer} buffer The buffer into which bytes are to be transferred.
     * @returns {Number} The total number of bytes read.
     */
    read : function(buffer) {
        var count = this._fd.recv(buffer.byteArray(), buffer.offset(), buffer.remaining());
        //jls.logger.logBuffer(jls.lang.Logger.WARN, buffer, 'Socket.read(), count: ' + count);
        if (count > 0) {
            buffer.incrementPosition(count);
        }
        return count;
    },
    /**
     * Writes a sequence of bytes to this channel from the given buffer.
     * 
     * @param {jls.lang.Buffer} buffer The buffer from which bytes are to be retrieved.
     * @returns {Number} The number of bytes written.
     */
    write : function(buffer) {
        //jls.logger.logBuffer(jls.lang.Logger.WARN, buffer, 'Socket.write()');
        var count = this._fd.send(buffer.byteArray(), buffer.offset(), buffer.remaining());
        if (count > 0) {
            buffer.incrementPosition(count);
        }
        return count;
    },
    /**
     * Adjusts this socket's blocking mode.
     * 
     * @param {Boolean} block Whether the socket is placed in blocking mode.
     * @returns {jls.net.Socket} This socket.
     */
    configureBlocking : function(block) {
        this._fd.configureBlocking(block);
        return this;
    },
    getChannel : function() {
        return this;
    },
    /*
     * Returns the native file descriptor object that represents this socket.
     * 
     * @returns {_native.net.Socket} the native file descriptor.
     */
    getFD : function() {
        return this._fd;
    }
});

