jls.loader.provide('jls.net.URL');

/**
 * @namespace Provides the classes for implementing networking applications.
 * @name jls.net
 */

jls.net.URL = jls.lang.Class.create( /** @lends jls.net.URL.prototype */
{
    /**
     * Creates an URL.
     *
     * @param {String} [protocol] The protocol or the URL as a string.
     * @param {String} [host] The host name.
     * @param {Number} [port] The port number.
     * @param {String} [file] The file part of the URL.
     * @constructs
	 * @class This class represents an Uniform Resource Locator.
	 * scheme://username:password@domain:port/path?query_string#anchor
     */
    initialize : function(protocol, host, port, file) {
        this._protocol = null;
        this._host = null;
        this._port = -1;
        this._path = null;
        this._query = null;
        this._ref = null;
        if (typeof protocol == 'string') {
            if (typeof host == 'undefined') {
                this._fromExternalForm(protocol);
            } else {
                this._protocol = protocol;
                this._host = host;
                this._port = port;
                this.setFile(file);
            }
        } else if (typeof protocol == 'object') {
            this._protocol = protocol.protocol;
            this._host = protocol.host;
            this._port = protocol.port;
            this.setFile(protocol.file);
        } else {
            throw new jls.lang.Exception('Invalid arguments');
        }
    },
    /**
     * Returns the protocol.
     * 
     * @returns {String} The protocol.
     */
    getProtocol : function() {
        return this._protocol;
    },
    /**
     * Returns the host.
     * 
     * @returns {String} The host.
     */
    getHost : function() {
        return this._host;
    },
    /**
     * Returns the port.
     * 
     * @returns {Number} The port.
     */
    getPort : function() {
        return this._port;
    },
    /**
     * Returns the default port associated protocol.
     * 
     * @returns {String} The default port associated protocol.
     */
    getDefaultPort : function() {
        if (this._protocol == null) {
            return -1;
        }
        var cname = this._protocol.toUpperCase() + '_PORT';
        if (cname in jls.net.URL) {
            return jls.net.URL[cname];
        }
        return -1;
    },
    /**
     * Returns the file.
     * 
     * @returns {String} The file.
     */
    getFile : function() {
        // URL:  <protocol>://<host>:<port>/<file>
        // file: <path>(#<ref>|?<query>)
        var file = this._path;
        if (this._query) {
            file += '?' + this._query;
        } else if (this._ref) {
            file += '#' + this._ref;
        }
        return file;
    },
    setFile : function(file) {
        var i = file.indexOf('?');
        if (i > 0) {
            this._path = file.substring(0, i);
            this._query = file.substr(i + 1);
            this._ref = null;
        } else {
            i = file.indexOf('#');
            if (i > 0) {
                this._path = file.substring(0, i);
                this._ref = file.substr(i + 1);
            } else {
                this._path = file;
                this._ref = null;
            }
            this._query = null;
        }
        return this;
    },
    /**
     * Returns the path.
     * 
     * @returns {String} The path.
     */
    getPath : function() {
        return this._path;
    },
    /**
     * Returns the query.
     * 
     * @returns {String} The query.
     */
    getQuery : function() {
        return this._query;
    },
    /**
     * Returns the anchor.
     * 
     * @returns {String} The anchor.
     */
    getRef : function() {
        return this._ref;
    },
    /**
     * Opens a connection.
     * 
     * @returns {Object} The connection.
     */
    openConnection : function() {
        // Look for a URLConnection class for the protocol
        var protocol = this._protocol.toLowerCase();
        if (! (protocol in jls.net.URL._urlConnectionClassMap)) {
            throw new jls.lang.Exception('No available URLConnection class for the protocol "' + protocol + '"');
        }
        var classname = jls.net.URL._urlConnectionClassMap[protocol];
        var clazz = jls.loader.require(classname);
        return new clazz(this);
    },
    _fromExternalForm : function(s) {
        var i = s.indexOf('://');
        if (i == -1) {
            throw new jls.lang.Exception('Invalid URL');
        }
        this._protocol = s.substring(0, i);
        i += 3;
        var j = s.indexOf('/', i);
        var file;
        if ((j > 0) && (j < s.length)) {
            file = s.substr(j);
        } else {
            file = '/';
        }
        var k = s.indexOf(':', i);
        if ((k != -1) && (k < j)) {
            this._host = s.substring(i, k);
            this._port = parseInt(s.substring(k + 1, j));
        } else {
            this._host = s.substring(i, j);
            this._port = -1;
        }
        this.setFile(file);
        return this;
    },
    /**
     * Returns the external form of the URL.
     * 
     * @returns {String} The URL as a string.
     */
    toExternalForm : function() {
        var s = this._protocol;
        s += '://' + this._host;
        if (this._port >= 0) {
            s += ':' + this._port;
        }
        s += this.getFile();
        return s;
    },
    /**
     * Returns the string value of the URL that is its external form.
     * 
     * @returns {String} The URL as a string.
     */
    toString : function() {
        return this.toExternalForm();
    }
});

Object.extend(jls.net.URL, /** @lends jls.net.URL */
{
    _urlConnectionClassMap : {
        'http' : 'jls.net.http.HttpURLConnection'
    },
    HTTP_PORT : 80,
    HTTPS_PORT : 443
});
