jls.loader.provide('jls.net.http.HttpHeader');

jls.loader.require('jls.lang.ByteBuffer');
jls.loader.require('jls.net.URL');
jls.loader.require('jls.net.SelectionHandler');

/*
see http://tools.ietf.org/html/rfc2068

HTTP-message    = Request | Response     ; HTTP/1.1 messages
generic-message = start-line
                 *message-header
                 CRLF
                 [ message-body ]
start-line      = Request-Line | Status-Line

If a Content-Length header field (section 14.14) is present, its
value in bytes represents the length of the message-body.

two classes request - response that can write - read http
a handler is called when read is done
*/

jls.net.http.HttpHeader = jls.lang.Class.create(jls.net.SelectionHandler, /** @lends jls.net.http.HttpHeader.prototype */
{
    /**
     * Creates an HTTP header.
     *
     * @param {String} [startLine] The start line.
     * @param {Object} [fields] The fields.
     * @constructs
	 * @augments jls.net.SelectionHandler
	 * @class This class represents an HTTP header.
     */
    initialize : function(startLine, fields) {
        this._startLine = '';
        this._fields = {};
        if (startLine) {
            this.setStartLine(startLine);
        }
        if (fields) {
            for (var k in fields) {
                this.setField(k, fields[k]);
            }
        }
        this._hdrStr = null;
        this._buffer = null;
    },
    clear : function() {
        this._startLine = '';
        this._fields = {};
    },
    getStartLine : function() {
        return this._startLine;
    },
    setStartLine : function(startLine) {
        this._startLine = startLine;
        return this;
    },
    hasField : function(key) {
        return key in this._fields;
    },
    getField : function(key) {
        return this._fields[key];
    },
    setField : function(key, value) {
        this._fields[key] = value;
        return this;
    },
    getContentLength : function() {
        if (this.hasField(jls.net.http.HttpHeader.HEADER_CONTENT_LENGTH)) {
            return parseInt(this.getField(jls.net.http.HttpHeader.HEADER_CONTENT_LENGTH), 10);
        } else {
            return 0;
        }
    },
    setContentLength : function(length) {
    	this.setField(jls.net.http.HttpHeader.HEADER_CONTENT_LENGTH, length);
    },
    /*length : function() {
        return this._buffer.capacity();
    },*/
    reset : function() {
        if (this._buffer != null) {
            this._buffer.clear();
        }
        return this;
    },
    /*
     * TODO Use jls.net.BufferSelectionHandler for R/W
     */
    onRead : function(input) {
        jls.logger.debug('onRead()');
        if (this._hdrStr == null) {
            this._hdrStr = '';
            // clear header
            this._startLine = '';
            this._fields = {};
        }
        var b = input.readByte();
        if (b < 0) {
            jls.logger.debug('b = ' + b);
            return -1; // cannot read
        }
        for (; b >= 0; b = input.readByte()) {
            jls.logger.debug('b = ' + b);
            this._hdrStr += String.fromCharCode(b);
            //jls.logger.debug('header part: ->' + this._hdrStr + '<-');
            if (this._hdrStr.substr(-4) == '\r\n\r\n') {
                this.fromString(this._hdrStr);
                this._hdrStr = null;
                return 1; // completed
            }
        }
        return 0; // in progress
    },
    onWrite : function(output) {
        if (this._buffer == null) {
            // TODO Correct invalid charset
            this._buffer = jls.lang.ByteBuffer.fromString(this.toString(), 'ASCII');
        }
        if (output.write(this._buffer) == 0) {
            return -1; // cannot write
        }
        if (this._buffer.remaining() == 0) {
            this._buffer.clear();
            return 1; // completed
        }
        return 0; // in progress
    },
    fromString : function(s) {
        var lines = s.split(jls.net.http.HttpHeader.lineSeparator);
        var startLine = lines[0];
        jls.logger.trace('set(parse) start line');
        this.setStartLine(startLine);
        for (var i = 1; i < lines.length; i++) {
            var entry = lines[i].split(/\s*:\s*/);
            if (entry.length != 2) {
                // TODO Throw something ?
                continue;
            }
            this.setField(entry[0], entry[1]);
        }
        return this;
    },
    toString : function() {
        var header = this.getStartLine() + jls.net.http.HttpHeader.lineSeparator;
        for (var k in this._fields) {
            header += k + ': ' + this._fields[k] + jls.net.http.HttpHeader.lineSeparator;
        }
        header += jls.net.http.HttpHeader.lineSeparator;
        return header;
    }
});

Object.extend(jls.net.http.HttpHeader, /** @lends jls.net.http.HttpHeader */
{
    fromString : function(s) {
        var hdr = s.startsWith(jls.net.http.HttpHeader.VERSION_PREFIX) ? new jls.net.http.HttpResponseHeader() : new jls.net.http.HttpRequestHeader();
        hdr.fromString(s);
        return hdr;
    },
    lineSeparator : '\r\n',
    //space : ' ',
/*
100 Continue
101 Switching Protocols
200 OK
201 Created
202 Accepted
203 Non-Authoritative Information
204 No Content
205 Reset Content
206 Partial Content
300 Multiple Choices
301 Moved Permanently
302 Moved Temporarily
303 See Other
304 Not Modified
305 Use Proxy
400 Bad Request
401 Unauthorized
402 Payment Required
403 Forbidden
404 Not Found
405 Method Not Allowed
406 Not Acceptable
407 Proxy Authentication Required
408 Request Time-out
409 Conflict
410 Gone
411 Length Required
412 Precondition Failed
413 Request Entity Too Large
414 Request-URI Too Large
415 Unsupported Media Type
500 Internal Server Error
501 Not Implemented
502 Bad Gateway
503 Service Unavailable
504 Gateway Time-out
505 HTTP Version not supported
*/
    HTTP_OK : 200,
    HTTP_NOT_FOUND : 404,

/*
OPTIONS
GET
HEAD
POST
PUT
DELETE
TRACE
*/
    METHOD_GET : 'GET',
    METHOD_POST : 'POST',

/*
There are two major versions, HTTP/1.0 that uses a separate connection for every document and
HTTP/1.1 that can reuse the same connection to download, for instance, images for the just served page.
Hence HTTP/1.1 may be faster as it takes time to set up such connections.
*/
    VERSION_PREFIX : 'HTTP/', // TODO Rename as Protocol?
    VERSION_1_0 : '1.0',
    VERSION_1_1 : '1.1',

    DEFAULT_USER_AGENT: 'JLS',

    HEADER_HOST: 'Host',
    HEADER_USER_AGENT: 'User-Agent',
    HEADER_ACCEPT: 'Accept',
    HEADER_ACCEPT_LANGUAGE: 'Accept-Language',
    HEADER_ACCEPT_ENCODING: 'Accept-Encoding',
    HEADER_ACCEPT_CHARSET: 'Accept-Charset',
    HEADER_KEEP_ALIVE: 'Keep-Alive',
    HEADER_PROXY_CONNECTION: 'Proxy-Connection',
    HEADER_COOKIE: 'Cookie',
    HEADER_CONTENT_LENGTH: 'Content-Length',
    HEADER_CONTENT_TYPE: 'Content-Type'
    
});

