jls.loader.provide('jls.net.http.HttpServer');

jls.loader.require('jls.lang.Thread');
jls.loader.require('jls.net.Socket');
jls.loader.require('jls.net.ServerSocket');
jls.loader.require('jls.net.SelectorThread');
jls.loader.require('jls.net.BufferSelectionHandler');
jls.loader.require('jls.net.http.HttpContext');
jls.loader.require('jls.net.http.HttpExchange');
jls.loader.require('jls.net.http.HttpRequestHeader');
jls.loader.require('jls.net.FileSelectionHandler');
jls.loader.require('jls.io.File');
jls.loader.require('jls.util.zip.ZipFile');

jls.net.http.HttpServer = jls.lang.Class.create(/** @lends jls.net.http.HttpServer.prototype */
{
    /**
     * Creates a HTTP server.
     *
     * @constructs
	 * @class A HTTP server.
     */
    initialize : function() {
        this._selectorThread = new jls.net.SelectorThread();
        this._contexts = {};
        this._defaultContext = new jls.net.http.HttpContext('');
        this._defaultContext.setHandler(jls.net.http.HttpServer.defaultHandler);
    },
    bind : function(addr, backlog) {
        jls.logger.info('Start HTTP server on ' + addr.toString());
        this._serverSocket = new jls.net.ServerSocket(addr.getPort(), backlog, addr.getAddress());
        this._serverSocket.configureBlocking(false);
        
        var key = this._selectorThread.register(this._serverSocket, jls.net.http.HttpServer._onAccept, jls.net.Selector.OP_ACCEPT);
        key.httpServer = this;
    },
    /**
     * Creates a context at the given path.
     * 
     * @param {String} path The path of the context.
     * @param {Function} [handler] The handler of the context.
     * @returns {jls.net.http.HttpContext} the created context.
     */
    createContext : function(path, handler)  {
        jls.logger.debug('HttpServer.createContext("' + path + '")');
		var context = new jls.net.http.HttpContext(path);
		if (handler) {
			context.setHandler(handler);
		}
        this._contexts[path] = context;
		return context;
    },
    getHttpContext : function(path)  {
        jls.logger.debug('getHttpContext("' + path + '")');
        var maxLength = -1;
        var context = null;
        for (var p in this._contexts) {
            jls.logger.trace('context "' + p + '"');
            if ((p.length > maxLength) && ((path.startsWith(p)) || (path + '/' == p))) {
				jls.logger.debug('Found matching context path "' + p + '" for "' + path + '"');
                maxLength = p.length;
                context = this._contexts[p];
            }
        }
        if (context == null) {
            jls.logger.debug('No matching context found for "' + path + '"');
            return this._defaultContext;
        }
        return context;
    },
    start : function() {
        this._selectorThread.start();
        jls.logger.info('HTTP server started');
    },
    stop : function() {
        this._selectorThread.stop();
        jls.logger.info('HTTP server stopped');
    }
});

Object.extend(jls.net.http.HttpServer, /** @lends jls.net.http.HttpServer */
{
    /**
     * Creates a HTTP server for a specified socket address.
     * 
     * @param {jls.net.InetSocketAddress} addr The socket address.
     * @param {Number} [backlog] The backlog.
     * @returns {jls.net.http.HttpServer} the created HTTP server.
     */
	create : function(addr, backlog) {
		var httpServer = new jls.net.http.HttpServer();
		if (addr) {
			httpServer.bind(addr, backlog);
		}
		return httpServer;
    },
    _onAccept : function(key, inFlags, outFlags) {
        jls.logger.info('Accept connection');
        
        var socket = key.socket.accept();
        socket.configureBlocking(false);
        
        var readKey = key.selectorThread.register(socket, jls.net.http.HttpServer._onReadHeader, jls.net.Selector.OP_READ);
        readKey.header = new jls.net.http.HttpRequestHeader();
        readKey.httpServer = key.httpServer;
    },
    _onClose : function(key) {
        jls.logger.info('Closing socket');
        key.selectorThread.lazyunregister(key);
        key.socket.close();
    },
    _onReadHeader : function(key, inFlags, outFlags) {
        var status = key.header.onRead(key.socket);
        switch (status) {
        case jls.net.SelectionHandler.STATUS_IN_PROGRESS:
            break;
        case jls.net.SelectionHandler.STATUS_DONE:
            // The header has been proceed, create the http exchange
            var context = key.httpServer.getHttpContext(key.header.getUri());
            var handle = context.getHandler();
            //jls.logger.debug('Request path is "' + context.getPath() + '"');
            //jls.logger.debug('header: ->' + key.header.toString() + '<-');
            key.httpExchange = new jls.net.http.HttpExchange(context, key.header, key.socket);
            // Call handler
            handle(key.httpExchange);
            // Dump response
            /*jls.lang.System.out.print('Response: ->');
            key.httpExchange.onWrite(jls.lang.System.out);
            jls.lang.System.out.println('<-');*/
            // Switch socket handler
            key.selectorThread.changeHandler(key, jls.net.http.HttpServer._onRead);
			// call it
			jls.net.http.HttpServer._onRead(key, inFlags, outFlags);
            break;
        default:
            jls.logger.warn('Invalid selection handler status (' + status + ')');
        case jls.net.SelectionHandler.STATUS_FAILURE:
            // No more data, closing socket
            jls.net.http.HttpServer._onClose(key);
            break;
        }
    },
    _onRead : function(key, inFlags, outFlags) {
        //jls.logger.trace('HttpServer._onRead()');
        var status = key.httpExchange.onRead(key.socket);
        switch (status) {
        case jls.net.SelectionHandler.STATUS_IN_PROGRESS:
            break;
        case jls.net.SelectionHandler.STATUS_DONE:
        	if (key.httpExchange._responseHandler != null) {
            	key.httpExchange._responseHandler.reset();
        	}
            key.selectorThread.changeHandler(key, jls.net.http.HttpServer._onWrite);
            key.selectorThread.changeMode(key, jls.net.Selector.OP_WRITE);
            break;
        default:
            jls.logger.warn('Invalid selection handler status (' + status + ')');
        case jls.net.SelectionHandler.STATUS_FAILURE:
            // No more data, closing socket
            jls.net.http.HttpServer._onClose(key);
            break;
        }
    },
    _onWrite : function(key, inFlags, outFlags) {
        //jls.logger.trace('HttpServer._onWrite()');
        var status = key.httpExchange.onWrite(key.socket);
        switch (status) {
        case jls.net.SelectionHandler.STATUS_IN_PROGRESS:
            break;
        case jls.net.SelectionHandler.STATUS_DONE:
            key.selectorThread.changeHandler(key, jls.net.http.HttpServer._onReadHeader);
            key.selectorThread.changeMode(key, jls.net.Selector.OP_READ);
            key.httpExchange = null;
            break;
        default:
            jls.logger.warn('Invalid selection handler status (' + status + ')');
        case jls.net.SelectionHandler.STATUS_FAILURE:
            // No more data, closing socket
            jls.net.http.HttpServer._onClose(key);
            break;
        }
    }
});

/*
 * Provide basic handler implementations.
 */
Object.extend(jls.net.http.HttpServer, /** @lends jls.net.http.HttpServer */
{
    /**
     * A shutdown handler.
     * 
     * @param {jls.net.http.HttpExchange} httpExchange The socket address.
     */
	shutdownHandler : function(httpExchange) {
        httpExchange.sendResponseHeaders(jls.net.http.HttpHeader.HTTP_OK, 'Shutdown in progress...');
        
        var thread = new jls.lang.Thread();
        thread.run = function() {
        	jls.lang.Thread.sleep(1000);
            jls.net.http.ServerMgr.getInstance().shutdown();
        };
        thread.start();
        //thread.join();
    },
    /**
     * A default handler implementation with the HTTP status: not found.
     * 
     * @param {jls.net.http.HttpExchange} httpExchange The socket address.
     */
    defaultHandler : function(httpExchange) {
        //jls.logger.trace('onRequest(' + httpExchange.getRequestURI() + ')');
        //httpExchange.setRequestBodySelectionHandler();
        httpExchange.sendResponseHeaders(jls.net.http.HttpHeader.HTTP_NOT_FOUND, 'Resource "' + httpExchange.getRequestURI() + '" not found.');
    },
    /**
     * A basic body handler implementation.
     * Parameters:<dl>
     * <dt>requestHandler</dt><dd>The function to call once the body has been read</dd>
     * </dl>
     * 
     * @param {jls.net.http.HttpExchange} httpExchange The socket address.
     */
    bodyHandler : function(httpExchange) {
		var requestHandler = httpExchange.getHttpContext().getAttribute('requestHandler');
        var contentLength = httpExchange.getRequestHeaders().getContentLength();
		// TODO use a buffer pool to reuse allocated buffers
        var buffer = jls.lang.ByteBuffer.allocate(contentLength);
        var sh = new jls.net.BufferSelectionHandler(buffer);
        var previousOnRead = sh.onRead;
        sh.onRead = function(channel) {
            var status = previousOnRead.call(sh, channel);
            if (status == jls.net.SelectionHandler.STATUS_DONE) {
                sh.buffer().flip();
                requestHandler(httpExchange, sh.buffer());
            }
            return status;
        };
        httpExchange.setRequestBodySelectionHandler(sh);
    },
    /**
     * A basic folder page generation implementation.
     * 
     * @param {String} reqPath The folder request path.
     * @param {Array} filenames The file name list.
     * @returns {String} The HTML content.
     */
    createFolderPage : function(reqPath, filenames) {
		if (! reqPath.endsWith('/')) {
			reqPath += '/';
		}
		var content = '<html><head></head><body>';
		for (var i = 0; i < filenames.length; i++) {
			content += '<a href="' + reqPath + filenames[i] + '">' + filenames[i] + '</a><br>';
		}
		content += '</body></html>';
		return content;
    },
    /**
     * Default content type objects.
     */
	defaultContentTypes : {
		'.html' : 'text/html; charset=UTF-8',
		'.js' : 'text/plain; charset=UTF-8',
		'.txt' : 'text/plain; charset=UTF-8',
		'.xml' : 'text/xml; charset=UTF-8'
	},
    /**
     * A basic file handler implementation.
     * Parameters:<dl>
     * <dt>rootFile</dt><dd>The root path to look for files</dd>
     * <dt>contentTypes</dt><dd>The extention / content type mapping to use</dd>
     * <dt>createFolderPage</dt><dd>A function to generate a folder page</dd>
     * </dl>
     * 
     * @param {jls.net.http.HttpExchange} httpExchange The socket address.
     */
    fileHandler : function(httpExchange) {
        var method = httpExchange.getRequestHeaders().getMethod();
		var reqPath = httpExchange.getRequestURI();
		var ctxPath = httpExchange.getHttpContext().getPath();
		var path = reqPath.substring(ctxPath.length);
        jls.logger.info('Handle ' + method + ' "' + path + '"');
        //jls.logger.debug('fileHandler(URI: ' + reqPath + ', ' + ctxPath + ', ' + path + ')');
        var indexFilename = httpExchange.getHttpContext().getAttribute('indexFilename');
        if (indexFilename == null) {
        	indexFilename = 'index.html';
        }
        var rootFile = httpExchange.getHttpContext().getAttribute('rootFile');
        if (rootFile == null) {
            throw new jls.lang.Exception('fileHandler: Root file not defined');
        }
        var contentTypes = httpExchange.getHttpContext().getAttribute('contentTypes');
        if (contentTypes == null) {
            contentTypes = {};
            httpExchange.getHttpContext().setAttribute('contentTypes', contentTypes);
        }
        var res = path.length == 0 ? rootFile : new jls.io.File(rootFile, path);
        //jls.logger.trace('Checking "' + res.getPath() + '"');
		if (res.exists()) {
			//jls.logger.info('Sending "' + res.getPath() + '"');
			if (indexFilename && res.isDirectory()) {
				var index = new jls.io.File(res, indexFilename);
				if (index.isFile()) {
					res = index;
				}
			}
			if (res.isFile()) {
                for (var ext in contentTypes) {
                    if (res.getPath().endsWith(ext)) {
                        httpExchange.getResponseHeaders().setField(jls.net.http.HttpHeader.HEADER_CONTENT_TYPE,
                            contentTypes[ext]);
                        break;
                    }
                }
				httpExchange.sendResponseHeaders(jls.net.http.HttpHeader.HTTP_OK, new jls.net.FileSelectionHandler(res));
			} else {
				var createFolderPage = httpExchange.getHttpContext().getAttribute('createFolderPage');
				if (createFolderPage != null) {
					var content = createFolderPage(reqPath, res.list());
					httpExchange.sendResponseHeaders(jls.net.http.HttpHeader.HTTP_OK, content);
				} else {
					//jls.logger.debug('Cannot browse folder');
					jls.net.http.HttpServer.defaultHandler(httpExchange);
				}
			}
		} else {
			//jls.logger.debug('Not found');
            jls.net.http.HttpServer.defaultHandler(httpExchange);
		}
    },
    zipHandler : function(httpExchange) {
        var method = httpExchange.getRequestHeaders().getMethod();
		var reqPath = httpExchange.getRequestURI();
		var ctxPath = httpExchange.getHttpContext().getPath();
		var path = reqPath.substring(ctxPath.length);
        jls.logger.info('Handle ' + method + ' "' + path + '"');
        //jls.logger.debug('fileHandler(URI: ' + reqPath + ', ' + ctxPath + ', ' + path + ')');
        
        var zipFile = httpExchange.getHttpContext().getAttribute('zipFile');
        if (zipFile == null) {
	        var file = httpExchange.getHttpContext().getAttribute('rootFile');
	        if (file == null) {
	            throw new jls.lang.Exception('zipHandler: Root file not defined');
	        }
	        jls.logger.info('Openning zip "' + file.getPath() + '"...');
	        zipFile = new jls.util.zip.ZipFile(file);
            httpExchange.getHttpContext().setAttribute('zipFile', zipFile);
        }
        var contentTypes = httpExchange.getHttpContext().getAttribute('contentTypes');
        if (contentTypes == null) {
            contentTypes = {};
            httpExchange.getHttpContext().setAttribute('contentTypes', contentTypes);
        }
        path = path.substring(1);
		if (zipFile.hasEntry(path)) {
	        var content = zipFile.getEntryContent(path);
            for (var ext in contentTypes) {
                if (path.endsWith(ext)) {
                    httpExchange.getResponseHeaders().setField(jls.net.http.HttpHeader.HEADER_CONTENT_TYPE,
                        contentTypes[ext]);
                    break;
                }
            }
            httpExchange.sendResponseHeaders(jls.net.http.HttpHeader.HTTP_OK, new jls.net.BufferSelectionHandler(content));
		} else {
			//jls.logger.debug('Not found');
            jls.net.http.HttpServer.defaultHandler(httpExchange);
		}
    }
});
