jls.loader.provide('jls.util.Image');

jls.loader.requireLibrary('jls_jpeg');

jls.loader.require('jls.io.File');
jls.loader.require('jls.io.FileChannel');

jls.util.Image = jls.lang.Class.create( /** @lends jls.util.Image.prototype */
{
    /**
     * Creates an image.
     *
     * @param {Number} width The image width.
     * @param {Number} height The image height.
     * @param {Number} [colorSpace] The color space.
     * @param {Number} [components] The component count per pixel.
     * @param {Number} [orientation] The image orientation.
     * @constructs
     * @class This class represents an image
     */
    initialize : function(width, height, colorSpace, components, orientation) {
        this.width = width;
        this.height = height;
        this.colorSpace = (typeof colorSpace == 'number') ? colorSpace : jls.util.Image.COLOR_SPACE_RGB;
        this.components = 0;
        switch (this.colorSpace) {
        case jls.util.Image.COLOR_SPACE_GRAYSCALE:
        	this.components = 1;
        	break;
        case jls.util.Image.COLOR_SPACE_RGB:
        case jls.util.Image.COLOR_SPACE_YCbCr:
        case jls.util.Image.COLOR_SPACE_BGR:
        	this.components = 3;
        	break;
        case jls.util.Image.COLOR_SPACE_CMYK:
        case jls.util.Image.COLOR_SPACE_YCCK:
        case jls.util.Image.COLOR_SPACE_BGRA:
        	this.components = 4;
        	break;
        case jls.util.Image.COLOR_SPACE_UNKNOWN:
        	break;
        default:
        	throw new jls.lang.Exception('Unknown color space (' + this.colorSpace + ')');
        }
        if (typeof components == 'number') {
        	this.components = components;
        }
        this.orientation = (typeof orientation == 'number') ? orientation : jls.util.Image.TOP_LEFT;
        this.bpp = this.components;
        this._bitPerPixel = this.bpp * 8;
        this.bpr = (this.width * this._bitPerPixel) / 8;
        if (this.bpr % 4 != 0) {
            this.bpr = this.bpr + 4 - this.bpr % 4;
        }
        this._size = this.bpr * this.height;
        jls.logger.info('jls.util.Image(' + width + ', ' + height + ', ' + colorSpace + ', ' + components + ', ' + orientation + ') ' + this._size + ' bytes');
        this._buffer = jls.lang.ByteBuffer.allocate(this._size);
    },
    /**
     * Returns the image width.
     * 
     * @returns {Number} The image width.
     */
    getWidth : function() {
        return this.width;
    },
    /**
     * Returns the image height.
     * 
     * @returns {Number} The image height.
     */
    getHeight : function() {
        return this.height;
    },
    /**
     * Returns the image buffer.
     * 
     * @returns {jls.lang.Buffer} The image buffer.
     */
    getBuffer : function() {
        return this._buffer;
    },
    _moveTo : function(x, y) {
        var offset = this.bpr * y + this.bpp * x;
        this._buffer.setPosition(offset);
        return this;
    },
    getPixel : function(x, y) {
        this._moveTo(x, y);
        var p = [];
        p.push(this._buffer.getByte());
        p.push(this._buffer.getByte());
        p.push(this._buffer.getByte());
        return p;
    },
    setPixel : function(x, y, r, g, b) {
        this._moveTo(x, y);
        this._buffer.putByte(r);
        this._buffer.putByte(g);
        this._buffer.putByte(b);
        return this;
    },
    /**
     * Returns the image orientation.
     * 
     * @returns {Number} The image orientation.
     */
    getOrientation : function() {
        return this.orientation;
    },
    setOrientation : function(orientation) {
        this.orientation = orientation;
    },
    getColorSpace : function() {
        return this.colorSpace;
    },
    changeColorSpace : function(colorSpace) {
    	if (this.colorSpace == colorSpace) {
    		return;
    	}
        switch (this.colorSpace) {
        case jls.util.Image.COLOR_SPACE_RGB:
            switch (colorSpace) {
            case jls.util.Image.COLOR_SPACE_YCbCr:
            	this.rgbToYuv();
            	break;
            case jls.util.Image.COLOR_SPACE_BGR:
            	this.componentSwap();
            	break;
            default:
            	throw new jls.lang.Exception('Unsupported color space change (' + this.colorSpace + '->' + colorSpace + ')');
            }
            break;
        case jls.util.Image.COLOR_SPACE_YCbCr:
            switch (colorSpace) {
            case jls.util.Image.COLOR_SPACE_RGB:
            	this.yuvToRgb();
            	break;
            default:
            	throw new jls.lang.Exception('Unsupported color space change (' + this.colorSpace + '->' + colorSpace + ')');
            }
            break;
        case jls.util.Image.COLOR_SPACE_BGR:
        	this.componentSwap();
        	break;
        default:
        	throw new jls.lang.Exception('Unsupported color space change (' + this.colorSpace + '->' + colorSpace + ')');
        }
        this.colorSpace = colorSpace;
    },
    getComponents : function() {
        return this.components;
    },
    /**
     * Returns a duplicate of this image.
     * 
     * @returns {jls.util.Image} The duplicate image.
     */
    duplicate : function() {
        jls.logger.info('jls.util.Image.duplicate()');
        var img = new jls.util.Image(this.getWidth(), this.getHeight(), this.getColorSpace(), this.getComponents(), this.getOrientation());
        //img2.putBuffer(img.getBuffer());
        img.getBuffer().byteArray().memcpy(0, this.getBuffer().byteArray(), 0, this.getBuffer().byteArray().size());
        return img;
    },
    /**
     * Returns a subsample of this image.
     * 
     * @param {Number} width The width of the subsample image.
     * @param {Number} height The height of the subsample image.
     * @returns {jls.util.Image} The subsample image.
     */
    subsample : function(width, height) {
        jls.logger.info('jls.util.Image.subsample(' + width + ', ' + height + ')');
        var img = new jls.util.Image(width, height, this.getColorSpace(), this.getComponents(), this.getOrientation());
        _native.jpeg.subsampleBilinear(this.getBuffer().byteArray(), this, img.getBuffer().byteArray(), img);
        return img;
    },
    /**
     * Swap the per pixel components depending on the indexes provided.
     * 
     * @param {Array} [indexes] The new indexes.
     */
    componentSwap : function(indexes) {
        jls.logger.info('jls.util.Image.componentSwap()');
    	if (! indexes) {
    		indexes = [];
    		for (var i = 0; i < this.bpp; i++) {
    			indexes.push(this.bpp - 1 - i);
    		}
    	}
        _native.jpeg.componentSwap(this.getBuffer().byteArray(), this, indexes);
    },
    /**
     * Applies kernel based convolution on this image.
     * 
     * @param {Array} kernel The kernel to use.
     * @param {Number} [componentStart] The first component to apply convolution.
     * @param {Number} [componentStop] The last component to apply convolution.
     */
    convolve : function(kernel, componentStart, componentStop) {
        jls.logger.info('jls.util.Image.convolve()');
        _native.jpeg.convolve(this.getBuffer().byteArray(), this, kernel, componentStart, componentStop);
    },
    /*
     * Merges an image with this image.
     * 
     * @param {jls.util.Image} img The image to merge with.
     */
    merge : function(img) {
        jls.logger.info('jls.util.Image.merge()');
        _native.jpeg.merge(this.getBuffer().byteArray(), this, img.getBuffer().byteArray(), img);
    },
    /**
     * Applies a matrix on the pixel components of this image.
     * 
     * @param {Array} matrix The matrix to use.
     * @param {Array} [delta] The delta to use.
     */
    componentMatrix : function(matrix, delta) {
        jls.logger.info('jls.util.Image.componentMatrix()');
    	if (! delta) {
    		delta = [];
    		for (var i = 0; i < this.bpp; i++) {
    			delta.push(0);
    		}
    	}
        _native.jpeg.componentMatrix(this.getBuffer().byteArray(), this, matrix, delta);
    },
    rgbToYuv : function() {
    	this.componentMatrix([
							0.29900, 0.58700, 0.11400,
							-0.16874, -0.33126, 0.50000,
							0.50000, -0.41869, -0.08131
    	                  ], [
    	                      0.5, 127.5, 127.5
    	                      //0, 128, 128
    	                      ]);
    },
    yuvToRgb : function() {
    	this.componentMatrix([
							1, 0, 1.402,
							1, -0.34414, -0.71414,
							1, 1.772, 0
    	                  ], [
    	                      -179.456 + 0.5, 44.04992 + 91.40992 - 0.5, -226.816 + 0.5
    	                      //-179.456, 44.04992 + 91.40992, -226.816
    	                      ]);
    }
});

/*
typedef struct tagBITMAPFILEHEADER {
  WORD  bfType; 2
  DWORD bfSize; 4
  WORD  bfReserved1; 2
  WORD  bfReserved2; 2
  DWORD bfOffBits; 4
} BITMAPFILEHEADER, *PBITMAPFILEHEADER; = 14

typedef struct tagBITMAPCOREHEADER {
  DWORD bcSize;
  WORD  bcWidth;
  WORD  bcHeight;
  WORD  bcPlanes;
  WORD  bcBitCount;
} BITMAPCOREHEADER, *PBITMAPCOREHEADER; = 12

typedef struct tagBITMAPINFOHEADER {
  DWORD biSize;
  LONG  biWidth;
  LONG  biHeight;
  WORD  biPlanes;
  WORD  biBitCount;
  DWORD biCompression;
  DWORD biSizeImage;
  LONG  biXPelsPerMeter;
  LONG  biYPelsPerMeter;
  DWORD biClrUsed;
  DWORD biClrImportant;
} BITMAPINFOHEADER, *PBITMAPINFOHEADER; = 9*4 + 2*2 = 40
*/

Object.extend(jls.util.Image, /** @lends jls.util.Image */
{
	/*
	 * Orientations from EXIF as firt row - first column.
	 * 
	 * Here is a description given by Adam M. Costello:
	 * For convenience, here is what the letter F would look like if it were tagged correctly and
	 * displayed by a program that ignores the orientation tag (thus showing the stored image):
	 * 
	 *   1        2       3      4         5            6           7          8
	 *
	 * 888888  888888      88  88      8888888888  88                  88  8888888888
	 * 88          88      88  88      88  88      88  88          88  88      88  88
	 * 8888      8888    8888  8888    88          8888888888  8888888888          88
	 * 88          88      88  88
	 * 88          88  888888  888888
	 * 
	 */
	TOP_LEFT:     1,
	TOP_RIGHT:    2,
	BOTTOM_RIGHT: 3,
	BOTTOM_LEFT:  4,
	LEFT_TOP:     5,
	RIGHT_TOP:    6,
	RIGHT_BOTTOM: 7,
	LEFT_BOTTOM:  8,
	/*
	 * Color spaces
	 */
	COLOR_SPACE_UNKNOWN:   _native.jpeg.COLOR_SPACE_UNKNOWN,
	COLOR_SPACE_GRAYSCALE: _native.jpeg.COLOR_SPACE_GRAYSCALE,
	COLOR_SPACE_RGB:       _native.jpeg.COLOR_SPACE_RGB,
	COLOR_SPACE_YCbCr:     _native.jpeg.COLOR_SPACE_YCbCr,
	COLOR_SPACE_CMYK:      _native.jpeg.COLOR_SPACE_CMYK,
	COLOR_SPACE_YCCK:      _native.jpeg.COLOR_SPACE_YCCK,
	
	COLOR_SPACE_BGR:       10,
	//COLOR_SPACE_BGRA:      11,
	
    /**
     * Reads a BMP.
     * 
     * @param {jls.io.FileInputStream} input The input to read the image.
     * @returns {jls.util.Image} The image.
     */
    readBMP : function(input) {
        /*var dumpObj = function(obj) {
            for (var key in obj) {
                if (typeof obj[key] == 'function') continue;
                jls.logger.debug('obj.' + key + ': ' + obj[key]);
            }
        };*/
        /*var dumpNum = function(num) {
            jls.logger.debug('num: ' + num + ' (0x' + num.toString(16) + ')');
        };*/
        var buffer = jls.lang.ByteBuffer.allocate(1024);
        buffer.setByteOrder(jls.lang.Buffer.LITTLE_ENDIAN);
        
        buffer.clear();
        buffer.setLimit(14);
        input.read(buffer);
        buffer.flip();
        //var typeStr = buffer.getString(2);
        //jls.logger.debug('type: "' + typeStr + '"');
        var typeStr = buffer.getShort();
        var size = buffer.getInt(true);
        //dumpNum(size);
        buffer.incrementPosition(4);
        var offBits = buffer.getInt(true);
        //dumpNum(offBits);

        buffer.clear();
        buffer.setLimit(offBits - 14);
        input.read(buffer);
        buffer.flip();
        
        var sSize = buffer.getInt(true);
        //dumpNum(sSize);
        var width, height, planes, bitCount;
        if (sSize == 12) {
            width = buffer.getShort(true);
            height = buffer.getShort(true);
            planes = buffer.getShort(true);
            bitCount = buffer.getShort(true);
        } else if (sSize == 40) {
            width = buffer.getInt(true);
            height = buffer.getInt(true);
            planes = buffer.getShort(true);
            bitCount = buffer.getShort(true);
        } else {
            throw 'Invalid BMP (' + sSize + ')';
        }
        jls.logger.debug('BMP: ' + width + ', ' + height + ', ' + planes + ', ' + bitCount);
        var img = new jls.util.Image(width, height);
        input.read(img.getBuffer());
        return img;
    },
    /**
     * Reads a JPEG.
     * 
     * @param {jls.io.FileInputStream} input The input to read the image.
     * @param {Number} [colorSpace] The output color space.
     * @param {Number} [gamma] The gamma.
     * @param {Function} [scaleCalculator] A callback to calculate the scale depending on the image dimension.
     * @returns {jls.util.Image} The image.
     */
    readJPEG : function(input, colorSpace, gamma, scaleCalculator) {
        var jdecompressor = new _native.jpeg.JpegDecompressor();
        var buffer = jls.lang.ByteBuffer.allocate(8192);
        /** @ignore */
        var srcMgr = {
            sourceBuffer : buffer,
            sourceByteArray : buffer.byteArray(),
            sourceCapacity : buffer.capacity(),
            sourceOffset : 0,
            initSource : function() {
                jls.logger.debug('initSource()');
            },
            fillSource : function() {
                jls.logger.trace('fillSource(), offset: ' + this.sourceOffset +
		        		', capacity: ' + this.sourceCapacity);
                this.sourceBuffer.clear();
                this.sourceBuffer.setPosition(this.sourceOffset);
                this.sourceCapacity = input.read(this.sourceBuffer);
            },
            skipSource : function(count) {
                jls.logger.warn('skipSource(' + count + '), offset: ' + this.sourceOffset +
		        		', capacity: ' + this.sourceCapacity);
            },
            termSource : function() {
                jls.logger.trace('termSource(), offset: ' + this.sourceOffset +
		        		', capacity: ' + this.sourceCapacity);
            }
        };
        var outColorSpace = _native.jpeg.COLOR_SPACE_UNKNOWN;
        if ((typeof colorSpace == 'number') && (colorSpace >= _native.jpeg.COLOR_SPACE_UNKNOWN) && (colorSpace <= _native.jpeg.COLOR_SPACE_YCCK)) {
        	outColorSpace = colorSpace;
        }
        gamma = gamma || 1.0;
        jdecompressor.initialize(srcMgr);
        if (outColorSpace != _native.jpeg.COLOR_SPACE_UNKNOWN) {
        	jdecompressor.setColorSpace(outColorSpace);
        }
        jdecompressor.setGamma(gamma);
        if (scaleCalculator) {
        	var s = scaleCalculator(jdecompressor.getWidth(true), jdecompressor.getHeight(true));
        	if (s) {
        		jdecompressor.setScale(s[0] /*Num*/, s[1] /*Denom*/);
        	}
        }
        jdecompressor.start();
        var width = jdecompressor.getWidth();
        var height = jdecompressor.getHeight();
        var components = jdecompressor.getComponents();
        jls.logger.debug('JPEG: ' + width + 'x' + height + ', ' + components + ', ' + outColorSpace);
        var img = new jls.util.Image(width, height);
        jdecompressor.read(img.getBuffer().byteArray(), -1, img.bpr);
        jdecompressor.finish();
        //img.componentSwap();
        img.colorSpace = jdecompressor.getColorSpace();
        img.setOrientation(jls.util.Image.BOTTOM_LEFT);
        if ((typeof colorSpace == 'number') && (colorSpace != _native.jpeg.COLOR_SPACE_UNKNOWN)) {
        	img.changeColorSpace(colorSpace);
        }
        buffer.free();
        return img;
    },
    /**
     * Writes a JPEG.
     * 
     * @param {jls.util.Image} img The image to write.
     * @param {jls.io.FileOutputStream} output The output to write the image.
     * @param {Number} [quality] The jpeg quality (0-100).
     */
    writeJPEG : function(img, output, quality) {
    	quality = quality || 75;
        var buffer = jls.lang.ByteBuffer.allocate(8192);
    	var jcompress = new _native.jpeg.JpegCompression();
    	var destMgr = {
                destinationBuffer : buffer,
                destinationByteArray : buffer.byteArray(),
    		    destinationCapacity : buffer.capacity(),
    		    destinationOffset : 0,
    		    initDestination : function() {
    		        jls.logger.trace('initDestination()');
    		    },
    		    flushDestination : function() {
    		        jls.logger.trace('flushDestination()');
                    this.destinationBuffer.clear();
                    output.write(this.destinationBuffer);
    		    },
    		    termDestination : function(count) {
    		        jls.logger.trace('termDestination(' + count + ')');
                    this.destinationBuffer.clear();
                    this.destinationBuffer.setLimit(count);
                    output.write(this.destinationBuffer);
    		    }
    		};
        jls.logger.debug('ColorSpace: ' + img.getColorSpace());
    	jcompress.start(destMgr, img, quality);
    	jcompress.write(img.getBuffer().byteArray(), -1);
    	jcompress.finish();
        buffer.free();
    }
});

