jls.loader.provide('jls.util.zip.ZipEntry');

jls.loader.require('jls.lang.Struct');

/**
 * @namespace Provides classes for reading ZIPped data.
 * @name jls.util.zip
 */

/*
 * specs from pkzip APPNOTE
 * All values are stored in little-endian byte order.
 * 0x06054b50 => 0x50, 0x4b, 0x05, 0x06
 */
jls.util.zip.ZipEntry = jls.lang.Class.create( /** @lends jls.util.zip.ZipEntry.prototype */
{
    /**
     * Creates a zip entry.
     *
     * @param {String} name The entry name.
     * @constructs
	 * @class This class represents a zip entry.
     */
    initialize : function(name, comment, extra, str) {
        this._name = name;
        this._comment = comment;
        this._extra = extra;
        this._compressedSize = 0;
        this._crc = null;
        this._method = null;
        this._size = 0;
        this._time = null;
        this._offset = 0;
        if ((typeof str != 'undefined') && (str instanceof jls.lang.Struct)) {
            this._compressedSize = str.get('compressedSize');
            this._crc = str.get('crc32');
            this._method = str.get('compressionMethod');
            this._size = str.get('uncompressedSize');
            /*
             * MS DOS Date & Time
             * bits: day(1 - 31), month(1 - 12), years(from 1980): 5, 4, 7 - second, minute, hour: 5, 6, 5
             */
            var time = str.get('lastModFileTime');
            var date = str.get('lastModFileDate');
            this._time = new Date();
            this._time.setUTCFullYear(1980 + ((date >>> 9) & 0x007f));
            this._time.setUTCMonth(((date >>> 5) & 0x000f) - 1);
            this._time.setUTCDate(date & 0x001f);
            this._time.setUTCHours((time >>> 11) & 0x001f);
            this._time.setUTCMinutes((time >>> 5) & 0x003f);
            this._time.setUTCSeconds(time & 0x001f);
            this._time.setUTCMilliseconds(0);
            jls.logger.debug('date: ' + this._time.toJSON());
            if (str.has('relativeOffset')) {
                this._offset = str.get('relativeOffset');
            }
        }
    },
    /**
     * Returns the compressed size.
     *
     * @returns {Number} The compressed size.
     */
    getCompressedSize : function() {
        return this._compressedSize;
    },
    /**
     * Returns the file comment.
     *
     * @returns {String} The file comment.
     */
    getComment : function() {
        return this._comment;
    },
    /**
     * Returns the CRC.
     *
     * @returns {Number} The CRC.
     */
    getCrc : function() {
        return this._crc;
    },
    getExtra : function() {
        return this._extra;
    },
    /**
     * Returns the compression method.
     *
     * @returns {Number} The compression method.
     */
    getMethod : function() {
        return this._method;
    },
    /**
     * Returns the file name.
     *
     * @returns {String} The file name.
     */
    getName : function() {
        return this._name;
    },
    /**
     * Returns the uncompressed size.
     *
     * @returns {Number} The uncompressed size.
     */
    getSize : function() {
        return this._size;
    },
    /**
     * Returns the last modification date.
     *
     * @returns {Date} The last modification date.
     */
    getTime : function() {
        return this._time;
    },
    getOffset : function() {
        return this._offset;
    }
});

Object.extend(jls.util.zip.ZipEntry, /** @lends jls.util.zip.ZipEntry */
{
    COMPRESSION_METHOD_STORED : 0,
    COMPRESSION_METHOD_DEFLATED : 8,
    GENERAL_PURPOSE_DEFLATE_MASK : 0x0006,
    GENERAL_PURPOSE_DEFLATE_NORMAL : 0x0000,
    GENERAL_PURPOSE_DEFLATE_MAXIMUM : 0x0002,
    GENERAL_PURPOSE_DEFLATE_FAST : 0x0004,
    GENERAL_PURPOSE_DEFLATE_SUPER_FAST : 0x0006,
    GENERAL_PURPOSE_DATA_DESCRIPTOR : 0x0008,
    GENERAL_PURPOSE_LANGUAGE_ENCODING : 0x0800,
    LOCAL_FILE_HEADER_SIGNATURE : 0x04034b50,
    FILE_HEADER_SIGNATURE : 0x02014b50,
    END_CENTRAL_DIR_SIGNATURE : 0x06054b50,
    getEndOfCentralDirectoryRecordStructDefinition : function() {
        return [
            {name: 'signature', type: 'UnsignedInt'},
            {name: 'diskNumber', type: 'UnsignedShort'},
            {name: 'centralDirectoryDiskNumber', type: 'UnsignedShort'},
            {name: 'diskEntryCount', type: 'UnsignedShort'},
            {name: 'entryCount', type: 'UnsignedShort'},
            {name: 'size', type: 'UnsignedInt'},
            {name: 'offset', type: 'UnsignedInt'},
            {name: 'commentLength', type: 'UnsignedShort'}
        ]; // 22
    },
    getDataDescriptorStructDefinition : function() {
        return [
            {name: 'crc32', type: 'UnsignedInt'},
            {name: 'compressedSize', type: 'UnsignedInt'},
            {name: 'uncompressedSize', type: 'UnsignedInt'}
        ]; // 12
    },
    getLocalFileHeaderStructDefinition : function() {
        return [
            {name: 'signature', type: 'UnsignedInt'},
            {name: 'versionNeeded', type: 'UnsignedShort'},
            {name: 'generalPurposeBitFlag', type: 'UnsignedShort'},
            {name: 'compressionMethod', type: 'UnsignedShort'},
            {name: 'lastModFileTime', type: 'UnsignedShort'},
            {name: 'lastModFileDate', type: 'UnsignedShort'},
            {name: 'crc32', type: 'UnsignedInt'},
            {name: 'compressedSize', type: 'UnsignedInt'},
            {name: 'uncompressedSize', type: 'UnsignedInt'},
            {name: 'filenameLength', type: 'UnsignedShort'},
            {name: 'extraFieldLength', type: 'UnsignedShort'}
        ]; // 30
    },
    getFileHeaderStructDefinition : function() {
        return [
            {name: 'signature', type: 'UnsignedInt'},
            {name: 'versionMadeBy', type: 'UnsignedShort'},
            {name: 'versionNeeded', type: 'UnsignedShort'},
            {name: 'generalPurposeBitFlag', type: 'UnsignedShort'},
            {name: 'compressionMethod', type: 'UnsignedShort'},
            {name: 'lastModFileTime', type: 'UnsignedShort'},
            {name: 'lastModFileDate', type: 'UnsignedShort'},
            {name: 'crc32', type: 'UnsignedInt'},
            {name: 'compressedSize', type: 'UnsignedInt'},
            {name: 'uncompressedSize', type: 'UnsignedInt'},
            {name: 'filenameLength', type: 'UnsignedShort'},
            {name: 'extraFieldLength', type: 'UnsignedShort'},
            {name: 'fileCommentLength', type: 'UnsignedShort'},
            {name: 'diskNumberStart', type: 'UnsignedShort'},
            {name: 'internalFileAttributes', type: 'UnsignedShort'},
            {name: 'externalFileAttributes', type: 'UnsignedInt'},
            {name: 'relativeOffset', type: 'UnsignedInt'}
        ]; // 46
    }
});
