jls.loader.provide('jls.util.zip.ZipFile');

jls.loader.require('jls.util.zip.ZipEntry');
jls.loader.require('jls.util.zip.Inflater');
jls.loader.require('jls.io.File');
jls.loader.require('jls.io.FileChannel');
jls.loader.require('jls.lang.Struct');

jls.util.zip.ZipFile = jls.lang.Class.create( /** @lends jls.util.zip.ZipFile.prototype */
{
    /**
     * Creates a zip file.
     *
     * @param {jls.io.File} file The file .
     * @constructs
	 * @class This class represents a zip file.
     */
    initialize : function(file) {
        this._buffer = jls.lang.ByteBuffer.allocate(1024);
        var buffer = jls.lang.ByteBuffer.allocate(64);
        buffer.setByteOrder(jls.lang.Buffer.LITTLE_ENDIAN);

        this._endOfCentralDirectoryRecord = new jls.lang.Struct(jls.util.zip.ZipEntry.getEndOfCentralDirectoryRecordStructDefinition(), buffer); // 22
        this._dataDescriptor = new jls.lang.Struct(jls.util.zip.ZipEntry.getDataDescriptorStructDefinition(), buffer); // 12
        this._localFileHeader = new jls.lang.Struct(jls.util.zip.ZipEntry.getLocalFileHeaderStructDefinition(), buffer); // 30
        this._fileHeader = new jls.lang.Struct(jls.util.zip.ZipEntry.getFileHeaderStructDefinition(), buffer); // 46

        this._file = file;
        this._channel = new jls.io.FileChannel(this._file);
        this._entryCount = 0;
        this._entries = {};
        
        try {
            //this._channel.seek(this._file.size() - this._endOfCentralDirectoryRecord.size());
            this._channel.seek(-this._endOfCentralDirectoryRecord.size(), jls.io.FileDescriptor.SEEK_END);

            this._readStruct(this._endOfCentralDirectoryRecord, jls.util.zip.ZipEntry.END_CENTRAL_DIR_SIGNATURE, 'End Of Central Directory Record');
            this._entryCount = this._endOfCentralDirectoryRecord.get('entryCount');
            jls.logger.debug('entry count: ' + this._entryCount);

            this._channel.seek(this._endOfCentralDirectoryRecord.get('offset'));
            for (var i = 0; i < this._entryCount; i++) {
                this._readStruct(this._fileHeader, jls.util.zip.ZipEntry.FILE_HEADER_SIGNATURE, 'File Header');
                var filename = this._readString(this._fileHeader.get('filenameLength'));
                this._channel.seek(this._fileHeader.get('extraFieldLength'), jls.io.FileDescriptor.SEEK_CUR);
                var comment = this._readString(this._fileHeader.get('fileCommentLength'));
                jls.logger.debug('entry: "' + filename + '"');
                this._entries[filename] = new jls.util.zip.ZipEntry(filename, comment, null, this._fileHeader);
            }
        }
        catch(e) {
            this.close();
            throw e;
        }
    },
    _readString : function(length) {
        this._buffer.clear();
        this._buffer.setLimit(length);
        this._channel.read(this._buffer);
        if (this._buffer.remaining() != 0) {
            throw new jls.lang.Exception('Unable to read string');
        }
        this._buffer.flip();
        return this._buffer.getString('UTF-8');
    },
    _readStruct : function(str, signature, name) {
        name = name || 'structure';
        str.clear();
        this._channel.read(str.buffer());
        if (str.buffer().remaining() != 0) {
            throw new jls.lang.Exception('Unable to read ' + name);
        }
        var sig = str.get('signature');
        jls.logger.debug('signature: 0x' + sig.toString(16));
        if (sig != signature) {
            throw new jls.lang.Exception('Invalid ' + name + ' signature (0x' + sig.toString(16) + ' != 0x' + signature.toString(16) + ')');
        }
    },
    /**
     * Closes the zip file.
     *
     */
    close : function() {
        this._channel.close();
        this._entryCount = 0;
        this._entries = {};
    },
    /**
     * Returns the zip entry count.
     *
     * @returns {Number} The zip entry count.
     */
    size : function() {
        return this._entryCount;
    },
    /**
     * Returns an entry depending on the specified name.
     *
     * @param {String} name The entry name.
     * @returns {jls.util.zip.ZipEntry} The zip entry or null.
     */
    getEntry : function(name) {
        if (name in this._entries) {
            return this._entries[name];
        }
        return null;
    },
    /**
     * Tells if this zip file contains a specified entry name.
     *
     * @param {String} name The entry name.
     * @returns {Boolean} true if this zip file contains a specified entry name.
     */
    hasEntry : function(name) {
        return name in this._entries;
    },
    /**
     * Returns a byte buffer containing the uncompressed entry content.
     *
     * @param {String} name The entry name.
     * @returns {jls.lang.ByteBuffer} The uncompressed entry content.
     */
    getEntryContent : function(name) {
        var entry = this.getEntry(name);
        if (entry == null) {
            throw new jls.lang.Exception('Entry not found');
        }
        if (entry.getMethod() != jls.util.zip.ZipEntry.COMPRESSION_METHOD_DEFLATED) {
            throw new jls.lang.Exception('Invalid method (' + entry.getMethod() + ')');
        }
        this._channel.seek(entry.getOffset(), jls.io.FileDescriptor.SEEK_SET);
        this._readStruct(this._localFileHeader, jls.util.zip.ZipEntry.LOCAL_FILE_HEADER_SIGNATURE, 'Local File Header');
        this._channel.seek(this._localFileHeader.get('filenameLength') + this._localFileHeader.get('extraFieldLength'), jls.io.FileDescriptor.SEEK_CUR);
        var compressedSize = this._localFileHeader.get('compressedSize');
        var uncompressedSize = this._localFileHeader.get('uncompressedSize');
        jls.logger.debug('compressedSize: ' + compressedSize);
        jls.logger.debug('uncompressedSize: ' + uncompressedSize);
        var inputBuffer = jls.lang.ByteBuffer.allocate(compressedSize);
        var outputBuffer = jls.lang.ByteBuffer.allocate(uncompressedSize);
        
        var inflater = new jls.util.zip.Inflater();
        this._channel.read(inputBuffer);
        inputBuffer.flip();
        inflater.setInput(inputBuffer);
        inflater.inflate(outputBuffer);
        inflater.end();
        
        outputBuffer.flip();
        inputBuffer.free();
        return outputBuffer;
    },
    getEntryContentAsString : function(name, csn) {
        var content = this.getEntryContent(name);
        return content.getString((typeof csn != 'undefined') ? csn : 'UTF-8');
    },
    getInputStream : function(entry) {
        return null;
    },
    /**
     * Returns the zip entries.
     *
     * @returns {Array} The zip entries.
     */
    entries : function() {
        var list = [];
        for (var name in this._entries) {
            list.push(this._entries[name])
        }
        return list;
    },
    /**
     * Returns the file name.
     *
     * @returns {String} The file name.
     */
    getName : function() {
        return this._file.getName();
    }
});

