jls.loader.provide('spyl.ffgui.FFmpeg');

jls.loader.require('jls.io.File');
jls.loader.require('jls.io.FileInputStream');
jls.loader.require('jls.io.InputStreamReader');
jls.loader.require('jls.io.FileOutputStream');
jls.loader.require('jls.io.OutputStreamWriter');
jls.loader.require('jls.io.BufferedReader');
jls.loader.require('jls.io.BufferChannel');
jls.loader.require('jls.lang.ProcessBuilder');
jls.loader.require('jls.lang.CharBuffer');
jls.loader.require('jls.io.Pipe');

/*
See http://www.transcoding.org/transcode?Aspect_Ratio
There are two types of aspect ratios involved in video editing.
One is display aspect ratio or DAR; this is the ratio most commonly referred to by the term "aspect ratio",
and is the ratio of the video frame's physical (displayed) width to its height,
regardless of the number of pixels used to represent the video image.
Typical DAR values are 4:3 for standard-definition video or 16:9 for widescreen television.

The other type of aspect ratio is pixel aspect ratio, or PAR (also known as "sample aspect ratio" or SAR).
This is the ratio of the width to the height of a single pixel in the video image;
a PAR of 1:1 means that each pixel is a perfect square,
while a PAR of 2:1 would mean that each pixel is a rectangle twice as wide as it is tall.
PAR can be used to refer either to the pixels in a video file, or to the pixels on a physical display device such as a television.

These two aspect ratios are related to each other and the number of pixels in the video frame (or display device) as follows:

    DAR   width
    --- = ------
    PAR   height
*/
/*
A few multimedia containers (MPEG-1, MPEG-2 PS, DV) allow to join video files by merely concatenating them. 

Hence you may concatenate your multimedia files by first transcoding them to these privileged formats, then using the humble cat command (or the equally humble copy under Windows), and finally transcoding back to your format of choice. 

ffmpeg -i input1.avi -sameq intermediate1.mpg
ffmpeg -i input2.avi -sameq intermediate2.mpg
cat intermediate1.mpg intermediate2.mpg > intermediate_all.mpg
ffmpeg -i intermediate_all.mpg -sameq output.avi

Notice that you should either use -sameq or set a reasonably high bitrate for your intermediate and output files, if you want to preserve video quality. 
*/
/*
Similarly, the yuv4mpegpipe format, and the raw video, raw audio codecs also allow concatenation, and the transcoding step is almost lossless. When using multiple yuv4mpegpipe(s), the first line needs to be discarded from all but the first stream. This can be accomplished by piping through tail as seen below. Note that when piping through tail you must use command grouping, { ;}, to background properly.
For example, let's say we want to join two FLV files into an output.flv file:

mkfifo temp1.a
mkfifo temp1.v
mkfifo temp2.a
mkfifo temp2.v
mkfifo all.a
mkfifo all.v
ffmpeg -i input1.flv -vn -f u16le -acodec pcm_s16le -ac 2 -ar 44100 - > temp1.a < /dev/null &
ffmpeg -i input2.flv -vn -f u16le -acodec pcm_s16le -ac 2 -ar 44100 - > temp2.a < /dev/null &
ffmpeg -i input1.flv -an -f yuv4mpegpipe - > temp1.v < /dev/null &
{ ffmpeg -i input2.flv -an -f yuv4mpegpipe - < /dev/null | tail -n +2 > temp2.v ; } &
cat temp1.a temp2.a > all.a &
cat temp1.v temp2.v > all.v &
ffmpeg -f u16le -acodec pcm_s16le -ac 2 -ar 44100 -i all.a \
       -f yuv4mpegpipe -i all.v \
       -sameq -y output.flv
rm temp[12].[av] all.[av]
*/

spyl.ffgui.FFmpeg = jls.lang.Class.create({
    initialize : function(home) {
    	this._home = home;
    	this._ffmpeg = spyl.ffgui.FFmpeg.getFileName(this._home, 'ffmpeg.exe');
    	this._ffprobe = spyl.ffgui.FFmpeg.getFileName(this._home, 'ffprobe.exe');
    },
    extractCodecs : function() {
    	var pargs = [this._ffmpeg, '-codecs'];
    	var pb = new jls.lang.ProcessBuilder(pargs);
    	var pipe = new jls.io.Pipe();
    	pb.setStdioRedirect(pipe.source(), jls.lang.ProcessBuilder.StandardOutput);
    	var process = pb.start();
    	var input = new jls.io.InputStreamReader(pipe.sink(), spyl.ffgui.FFmpeg.charset);
        var reader = new jls.io.BufferedReader(input);
        var codecs = {
        		audio: {
        			decoder: {},
        			encoder: {}
        		},
        		video: {
        			decoder: {},
        			encoder: {}
        		}
        };
		/*
		Codecs:
		 D..... = Decoding supported
		 .E.... = Encoding supported
		 ..V... = Video codec
		 ..A... = Audio codec
		 ..S... = Subtitle codec
		 ...S.. = Supports draw_horiz_band
		 ....D. = Supports direct rendering method 1
		 .....T = Supports weird frame truncation
		 ------
		*/
        var header = true;
        for (;;) {
            var line = reader.readLine();
            if (header) {
            	if (line == ' ------') {
            		header = false;
            	}
            	continue;
            }
            if ((line == null) || (line.length == 0)) {
            	break;
            }
            var codecParts = line.match(/ ([D ][E ][VAS ][S ][D ][T ]) ([^ ]+) +(.*)/);
            if ((codecParts == null) || (codecParts.length != 4)) {
                throw new jls.lang.Exception('Invalid line "' + line + '"');
            }
            var cap = codecParts[1];
            var type;
            if (cap.charAt(2) == 'A') {
            	type = 'audio';
            } else if (cap.charAt(2) == 'V') {
            	type = 'video';
            } else {
            	continue; // skip
            }
            var name = codecParts[2];
            var description = codecParts[3];
            if (cap.charAt(0) == 'D') {
            	codecs[type].decoder[name] = description;
            }
            if (cap.charAt(1) == 'E') {
            	codecs[type].encoder[name] = description;
            }
        }
    	var exitCode = process.waitFor();
    	return codecs;
    },
    extractFormats : function() {
    	var pargs = [this._ffmpeg, '-formats'];
    	var pb = new jls.lang.ProcessBuilder(pargs);
    	var pipe = new jls.io.Pipe();
    	pb.setStdioRedirect(pipe.source(), jls.lang.ProcessBuilder.StandardOutput);
    	var process = pb.start();
    	process.setExitCallback(function() {
        	//jls.logger.warn('onExit');
        	var buffer = jls.lang.ByteBuffer.fromString(jls.io.BufferedReader.separator + 'yo' + jls.io.BufferedReader.separator);
        	pipe.source().write(buffer);
        });
    	var input = new jls.io.InputStreamReader(pipe.sink(), spyl.ffgui.FFmpeg.charset);
    	var reader = new jls.io.BufferedReader(input);
        var formats = {
    			decoder: {},
    			encoder: {}
		};
		/*
		File formats:
		 D. = Demuxing supported
		 .E = Muxing supported
		 --
		*/
        var header = true;
        for (;;) {
            var line = reader.readLine();
            if (header) {
            	if (line == ' --') {
            		header = false;
            	}
            	continue;
            }
            if ((line == null) || (line.length == 0)) {
            	break;
            }
            var formatParts = line.match(/ ([D ][E ]) ([^ ]+) +(.*)/);
            if ((formatParts == null) || (formatParts.length != 4)) {
                throw new jls.lang.Exception('Invalid line "' + line + '"');
            }
            var cap = formatParts[1];
            var name = formatParts[2];
            var description = formatParts[3];
            if (cap.charAt(0) == 'D') {
            	formats.decoder[name] = description;
            }
            if (cap.charAt(1) == 'E') {
            	formats.encoder[name] = description;
            }
        }
    	return formats;
    },
    probe : function(filename) {
    	// -pretty -show_format -show_streams
    	var pargs = [this._ffprobe, '-show_format', '-show_streams', filename];
    	var pb = new jls.lang.ProcessBuilder(pargs);
    	var pipe = new jls.io.Pipe();
    	pb.setStdioRedirect(pipe.source(), jls.lang.ProcessBuilder.StandardOutput);
    	var process = pb.start();
    	var input = new jls.io.InputStreamReader(pipe.sink(), spyl.ffgui.FFmpeg.charset);
        var reader = new jls.io.BufferedReader(input);
        var pr = {
            streams: []
        };
        var scope = null;
        for (;;) {
            var line = reader.readLine();
            //jls.lang.System.out.println('-->' + line + '<--');
            if (line.indexOf('[') == 0) {
                var block = line.match(/\[(\/?)(.*)\]/);
                if ((block == null) || (block.length != 3)) {
                    throw new jls.lang.Exception('Invalid line "' + line + '"');
                }
                //jls.lang.System.out.println('block: [' + block.join(', ') + ']');
                if (block[2] == 'STREAM') {
                    if (block[1] == '/') {
                        scope = null;
                    } else {
                        pr.streams.push({});
                        scope = pr.streams[pr.streams.length - 1];
                    }
                } else if (block[2] == 'FORMAT') {
                    if (block[1] == '/') {
                        break;
                    }
                    scope = pr;
                } else {
                    throw new jls.lang.Exception('Invalid section name "' + block[2] + '"');
                }
            } else {
                var keyValue = line.match(/([a-zA-Z0-9\-_]+)=(.*)/);
                if ((keyValue == null) || (keyValue.length != 3)) {
                    throw new jls.lang.Exception('Invalid line "' + line + '"');
                    break;
                }
                //jls.lang.System.out.println('keyValue: [' + keyValue.join(', ') + ']');
                scope[keyValue[1]] = keyValue[2];
            }
        }
        pr.streamByCodecType = {};
        for (var i = 0; i < pr.streams.length; i++) {
            var stream = pr.streams[i];
            if (! ('codec_type' in stream)) {
                continue;
            }
            var codecType = stream['codec_type'];
            if (! (codecType in pr.streamByCodecType)) {
                pr.streamByCodecType[codecType] = [];
            }
            pr.streamByCodecType[codecType].push(stream);
        }
    	var exitCode = process.waitFor();
    	return pr;
    },
    extractFrame : function(srcFilename, destFilename) {
    	var process = this.start(destFilename, ['-f', 'rawvideo', '-vcodec', 'bmp', '-vframes', '1', '-an'],
    			srcFilename, ['-ss', '00:00:01'], ['-loglevel', 'quiet']);
    	return process.waitFor();
    },
    computeArguments : function(destFilename, destOptions, srcFilename, srcOptions, globalOptions) {
    	var args = [this._ffmpeg];
        if (globalOptions) {
            args = args.concat(globalOptions);
        }
        if (srcOptions) {
            args = args.concat(srcOptions);
        }
        if (srcFilename) {
            args = args.concat(['-i', srcFilename]);
        }
        if (destOptions) {
            args = args.concat(destOptions);
        }
        if (destFilename) {
            args = args.concat(['-y', destFilename]);
        }
        return args;
    },
    start : function(destFilename, destOptions, srcFilename, srcOptions, globalOptions) {
    	var args = this.computeArguments(destFilename, destOptions, srcFilename, srcOptions, globalOptions);
        jls.lang.System.out.println('ffmpeg arguments: ' + args.join(' '));
    	//return new jls.lang.Process(args);
        
    	var pb = new jls.lang.ProcessBuilder(args);
    	var pipe = new jls.io.Pipe();
    	//pb.setStdioRedirect(pipe.source(), jls.lang.ProcessBuilder.StandardOutput);
    	pb.setStdioRedirect(pipe.source(), jls.lang.ProcessBuilder.StandardError);
    	var process = pb.start();

        var thread = new jls.lang.Thread(true); // TODO Remove
        thread.run = function() {
        	var buffer = jls.lang.ByteBuffer.allocate(1024);
        	for (;;) {
        		buffer.clear();
            	pipe.sink().read(buffer);
            	buffer.flip();
            	jls.lang.System.out.getOutputStream().write(buffer);
        	}
        };
        jls.logger.debug('Start thread');
        thread.start();
    	
    	return process;
    }
});

Object.extend(spyl.ffgui.FFmpeg,
{
	charset : 'ASCII', // Should be Cp850 or UTF-8 ?
	getFileName : function(dir, filename) {
        var file = new jls.io.File(dir, filename);
        if (! file.isFile()) {
            throw new jls.lang.Exception('Cannot found "' + file.getPath() + '"');
        }
        return file.getPath();
    }
});

