define([
  'jls/lang/Class',
  'jls/lang/Exception',
  'jls/io/cs/CharDecoder',
  'jls/io/cs/CharEncoder'
], function (
  Class,
  Exception,
  CharDecoder,
  CharEncoder
) {

var Charset;
/**
 * @namespace Provides character sets.
 * @see jls.io.cs.Charset
 * @name jls.io.cs
 */

/**
 * @class This class represents a character set and the associated codec.
 * The default charsets are Cp1252, Cp850, ISO-8859-1, UTF-8 and ASCII.
 * @name jls.io.cs.Charset
 */
Charset = Class.create( /** @lends jls.io.cs.Charset.prototype */
{
    /**
     * Creates a charset.
     * 
     * @private
     */
    initialize : function(name) {
        this._name = name;
    },
    /**
     * Returns the charset name.
     *
     * @returns {String} The charset name.
     */
    getName : function() {
        return this._name;
    },
    /**
     * Decodes byte buffer and returns the string.
     *
     * @param {jls.lang.Buffer} buf The buffer to decode.
     * @returns {String} The decoded string.
     */
    decode : function(buf) {
        return '';
    },
    /**
     * Encodes a string and returns a buffer.
     *
     * @param {String} str The string to encode.
     * @returns {jls.lang.Buffer} The buffer.
     */
    encode : function(str) {
        return null;
    },
    /**
     * Creates a new decoder for this charset.
     *
     * @returns {jls.io.cs.CharDecoder} The buffer.
     */
    newDecoder : function() {
        return new CharDecoder(this);
    },
    /**
     * Creates a new encoder for this charset.
     *
     * @returns {jls.io.cs.CharEncoder} The buffer.
     */
    newEncoder : function() {
        return new CharEncoder(this);
    }
});


Object.extend(Charset, /** @lends jls.io.cs.Charset */
{
	_defaultCharset : null,
	_charsetMapping : {
		'Cp1252' : 'jls/io/cs/Cp1252',
		'Cp850' : 'jls/io/cs/Cp850',
		'ISO-8859-1' : 'jls/io/cs/ISO_8859_1',
		'UTF-8' : 'jls/io/cs/UTF_8',
		'ASCII' : 'jls/io/cs/Charset'
	},
	_availableCharsets : {},
    /**
     * Returns available charsets.
     * 
     * @returns {Array} The available charsets.
     */
	availableCharsets : function() {
        return Object.values(Charset._availableCharsets);
    },
	addCharset : function(cs) {
		Charset._availableCharsets[cs.getName()] = cs;
    },
    /**
     * Returns the default charset.
     * 
     * @returns {jls.io.cs.Charset} The default charset.
     */
    defaultCharset : function() {
    	if (Charset._defaultCharset == null) {
            var csn = _native.core.properties['file.encoding'];
    		if (csn && Charset.isSupported(csn)) {
    			Charset._defaultCharset = Charset.forName(csn);
    		} else {
        		Charset._defaultCharset = Charset.forName('ASCII');
    		}
    	}
    	return Charset._defaultCharset;
    },
    /**
     * Tells if the specified charset is supported.
     * 
     * @param {String} name The charset name.
     * @returns {Boolean} true if the specified charset is supported.
     */
    isSupported : function(name) {
    	if (typeof name != 'string') {
			throw new Exception('Invalid charset name');
    	}
    	if (name in Charset._availableCharsets) {
			return true;
    	}
    	if (name in Charset._charsetMapping) {
            var cs = require(Charset._charsetMapping[name]);
			return name in Charset._availableCharsets;
    	}
		return false;
    },
    /**
     * Returns the specified charset.
     * 
     * @param {String} name The charset name.
     * @returns {jls.io.cs.Charset} The charset.
     */
	forName : function(name) {
		if (! Charset.isSupported(name)) {
			throw new Exception('Unsupported charset "' + name + '"');
		}
        return Charset._availableCharsets[name];
    }
});

// static
Charset.addCharset(new Charset('ASCII'));



return Charset;
});
