define(['jls/lang/Class', 'jls/lang/AssertionError'], function (Class, AssertionError) {

var Assert;
/**
 * @class A set of assert methods. Messages are only displayed when an assert fails.
 * @name jls.jsunit.Assert
 */
Assert = Class.create({});

Object.extend(Assert, /** @lends jls.jsunit.Assert */
{
    /**
     * Fails a test with the given message.
     * 
     * @param {String} [message] The message associated with the failure.
     */
    fail : function(message) {
        throw new AssertionError(message);
    },
    failNotEquals : function(expected, actual, message) {
        var msg = (typeof message == 'string') ? (message + ' ') : '';
        Assert.fail(msg + 'expected:<' + expected + '> but was:<' + actual + '>');
    },
    failNotSame : function(expected, actual, message) {
        var msg = (typeof message == 'string') ? (message + ' ') : '';
        Assert.fail(msg + 'expected same:<' + expected + '> was not:<' + actual + '>');
    },
    failSame : function(message) {
        var msg = (typeof message == 'string') ? (message + ' ') : '';
        Assert.fail(msg + 'expected not same');
    },
    /**
     * Asserts that a condition is true.
     * 
     * @param {Boolean} condition The condition to verify.
     * @param {String} [message] The message associated with the assertion.
     */
    assertTrue : function(condition, message) {
        if (! condition) {
            Assert.fail(message);
        }
    },
    /**
     * Asserts that a condition is false.
     * 
     * @param {Boolean} condition The condition to verify.
     * @param {String} [message] The message associated with the assertion.
     */
    assertFalse : function(condition, message) {
        if (condition) {
            Assert.fail(message);
        }
    },
    /**
     * Asserts that a value is not null.
     * 
     * @param {Object} value The value to verify.
     * @param {String} [message] The message associated with the assertion.
     */
    assertNotNull : function(value, message) {
        if (value == null) {
            Assert.fail(message);
        }
    },
    /**
     * Asserts that two values differ.
     * 
     * @param {Object} expected The expected value.
     * @param {Object} actual The actual value.
     * @param {String} [message] The message associated with the assertion.
     */
    assertNotSame : function(expected, actual, message) {
        if (expected === actual) {
            Assert.failSame(message);
        }
    },
    /**
     * Asserts that a value is null.
     * 
     * @param {Object} value The value to verify.
     * @param {String} [message] The message associated with the assertion.
     */
    assertNull : function(value, message) {
        if (value != null) {
            Assert.fail(message);
        }
    },
    /**
     * Asserts that two values are the same.
     * 
     * @param {Object} expected The expected value.
     * @param {Object} actual The actual value.
     * @param {String} [message] The message associated with the assertion.
     */
    assertSame : function(expected, actual, message) {
        if (expected !== actual) {
            Assert.failNotSame(expected, actual, message);
        }
    },
    /**
     * Asserts that two values are equal.
     * 
     * @param {Object} expected The expected value.
     * @param {Object} actual The actual value.
     * @param {String} [message] The message associated with the assertion.
     */
    assertEquals : function(expected, actual, message) {
        if (expected != actual) {
            Assert.failNotEquals(expected, actual, message);
        }
    }
});

return Assert;
});
