define([
  'jls/lang/Class',
  'jls/lang/Logger',
  'jls/gui/GuiUtilities',
  'jls/gui/Button',
  'jls/gui/Edit',
  'jls/gui/Label',
  'jls/html/DomEvent',
  'jls/html/HtmlElement',
  'jls/html/Table',
  'jls/util/Cookie',
  'jls/util/EventBroker',
  'jls/util/Formatter',
  'jls/util/Location',
  'jls/util/Resource',
  'jls/util/Sxe',
  'jls/util/SxeEnvelope',
  'jls/util/StringCodec',
  'jls/util/XMLHttpRequest',
  'jls/security/MessageDigest',
  'spyl/pb/MessageBox',
  'spyl/pb/Comment',
  'spyl/pb/Context',
  'spyl/pb/Group',
  'spyl/pb/GroupPreview',
  'spyl/pb/Icon',
  'spyl/pb/Media',
  'spyl/pb/MovieElement',
  'spyl/pb/PictureElement',
  'spyl/pb/PreviewElement',
  'spyl/pb/User'
], function (
  Class,
  Logger,
  GuiUtilities,
  Button,
  Edit,
  Label,
  DomEvent,
  HtmlElement,
  Table,
  Cookie,
  EventBroker,
  Formatter,
  Location,
  Resource,
  Sxe,
  SxeEnvelope,
  StringCodec,
  XMLHttpRequest,
  MessageDigest,
  MessageBox,
  Comment,
  Context,
  Group,
  GroupPreview,
  Icon,
  Media,
  MovieElement,
  PictureElement,
  PreviewElement,
  User
) {

var Launcher;
/*
 * TODO
 * (Auto-)Refresh the comment board
 * Poll for new comments/medias
 * Start on new medias if any
 * Add tags
 */
Launcher = Class.create({
    initialize: function() {
        this.broker = new EventBroker();
        this.sxe = new Sxe('script/default.php');
        this.medias = [];
        this.root = null;
        this.parent = GuiUtilities.getRoot();
        this.contextHistory = new Array();
        this.contextHistoryIndex = -1;
        this.resource = null;
        this.adminEMail = null;

        this.users = {};
        this.user = null;
        this.userId = Cookie.get('userId');
        this.password = Cookie.get('password');
        this.rememberMe = false;
        Logger.getInstance().info('Cookie says: User Id: ' + this.userId);
        
        this.connected = false;
        this.logon = null;
        this.menu = null;
        this.comment = null;

        this.broker.subscribeAll([
              'onOpenMenu',
              'onCloseMenu',
              'onLogoff',
              'onRegister',
              'onDisplayInfo',
              'onAuthenticated',
              'onAccessGranted',
              'onLogon',
              'onMediasLoaded',
              'onConnectOk',
              'onLostPassword',
              'onManageAccount',
              
              'onOpenGroup',
              'onOpenMedia',
              'onDisplayMedia',
              'onDisplayBoard',
              'onDisplayBoardFull',
              'onDisplayGroup',
              'onGoPrevious',
              'onGoNext',
              'onGoHome',
              'onGoUp',
              'onGoBackward',
              'onGoForward',
              'onComment',
              
              'onDisplayUsers',
              'onFeedUsers'
          ], this, this);
    },
    launch: function() {
        Logger.getInstance().trace('launch()');
        this.resource = Resource.load('spyl/pb/client');
        this.dateFormat = this.resource.get('date.format');
        this.commentLabel = this.resource.get('comment');
        this.commentsLabel = this.resource.get('comments');
        this.parent.setResource(this.resource);
        this.parent.setEventBroker(this.broker);
        this.createElements();
        HtmlElement.addEventListener(document, 'keydown', (function(e) {
            this.onKey.call(this, new DomEvent(e, this));
        }).bind(this));

        if (this.userId && this.password) {
            this.broker.publish('onLogon');
        }
        // check url
        var hash = Location.getHash();
        if (hash) {
            Location.setHash();
        }
        var query = Location.parseQuery(hash);
        if ((query != null) && (query.operation)) {
            this.onQuery(query);
        }
        Logger.getInstance().trace('done');
    },
    onKey: function(event) {
        var keyCode = event.keyCode();
        //Logger.getInstance().debug('onKey(): -->' + keyCode + '<--');
        if ((this.currentContext == null) || this.comment) {
            return;
        }
        switch (keyCode) {
        case 38: // up
            switch (this.currentContext.viewName) {
            case 'media':
                this.broker.publish('onDisplayBoard', event);
                break;
            case 'board':
                this.broker.publish('onDisplayGroup', event);
                break;
            case 'group':
                event.message.preventDefault();
                if (this.currentContext.group != this.root) {
                    this.changeContext(new Context('group', this.root, 0));
                }
                break;
            }
            break;
        case 40: // down
            switch (this.currentContext.viewName) {
            /*case 'media':
                event.preventDefault();
                break;*/
            case 'board':
                this.broker.publish('onDisplayMedia', event);
                break;
            case 'group':
                this.broker.publish('onDisplayBoard', event);
                break;
            }
            break;
        }
        if (this.currentContext.viewName != 'media') {
            return;
        }
        switch (keyCode) {
        case 37: // left
            this.broker.publish('onGoBackward', event);
            break;
        case 39: // right
            this.broker.publish('onGoForward', event);
            break;
        }
    },
    onQuery: function(query) {
        var envelope = new SxeEnvelope();
        envelope.addRequest({name: query.operation, attributes: query});
        this.sxe.send(envelope, (function(response) {
            var envelope = response.getEnvelope();
            Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
            var result = envelope.getResult(0);
            var msg = null;
            if (result.getName() == 'failure') {
                msg = this.resource.getf('operation.failure', query.operation, result.getText());
            } else {
                switch (query.operation) {
                case 'activateUser':
                    if (result.getName() == 'success') {
                        msg = this.resource.get('user.activated');
                    }
                    break;
                }
            }
            this.broker.publish('onDisplayInfo', msg || this.resource.get('error'));
        }).bind(this));
    },
    createElements: function() {
        Logger.getInstance().trace('createElements()');
        this.viewport = new HtmlElement({attributes: {htmlTagName: 'div', 'class': 'viewport'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'table', 'class': 'border'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'tbody'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'tr'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_nw'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_n'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_ne'}}
            ]},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'tr', id: 'bordered_fix_row'}, style: {display: 'none'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_w'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'bordered_fix', id: 'bordered_fix'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_e'}}
            ]},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'tr', id: 'bordered_auto_row'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_w'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'bordered_auto', id: 'bordered_auto'}, children: [
            {classname: 'jls/gui/Label', attributes: {text: '#welcome'}}
            ]},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_e'}}
            ]},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'tr'}, children: [
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_sw'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_s'}},
            {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'td', 'class': 'border_se'}}
            ]}]}]}]},
            this.parent);
        this.toolbar = new HtmlElement({attributes: {htmlTagName: 'div'},
            style: {position: 'absolute', textAlign: 'left', left: '3px', top: '0px', display: 'none'}, children: [
            {classname: 'jls/html/Table', attributes: {fixedSize: 3}, children: [
                   {classname: 'spyl/pb/Icon', attributes: {id: 'groupIcon', iconSrc: 'style/group.png', iconTitle: '#icon.group'}, events: {click: 'onDisplayGroup'}},
                   {classname: 'spyl/pb/Icon', attributes: {id: 'boardIcon', iconSrc: 'style/board.png', iconTitle: '#icon.board'}, events: {click: 'onDisplayBoard'}},
                   {classname: 'spyl/pb/Icon', attributes: {id: 'mediaIcon', iconSrc: 'style/media.png', iconTitle: '#icon.media'}, events: {click: 'onDisplayMedia'}}
           ]},
           {classname: 'jls/html/Table', attributes: {id: 'mediaToolbar', fixedSize: 3}, children: [
                   {classname: 'spyl/pb/Icon', attributes: {id: 'commentIcon', iconSrc: 'style/list-add.png', iconTitle: '#icon.media.addComment'}, events: {click: 'onComment'}},
                   {classname: 'spyl/pb/Icon', attributes: {id: 'backwardIcon', iconSrc: 'style/media-skip-backward.png', iconTitle: '#icon.media.backward'}, events: {click: 'onGoBackward'}},
                   {classname: 'spyl/pb/Icon', attributes: {id: 'forwardIcon', iconSrc: 'style/media-skip-forward.png', iconTitle: '#icon.media.forward'}, events: {click: 'onGoForward'}}
           ]}
        ]}, this.parent);
        this.header = new HtmlElement({attributes: {htmlTagName: 'div'},
           style: {position: 'absolute', textAlign: 'right', right: '20px', top: '0px'}, children: [
              {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', id: 'connectLink', href: '#', textContent: '#logon'}, events: {click: 'onOpenMenu'}}
          ]},
          this.parent);
        this.logon = new HtmlElement({attributes: {htmlTagName: 'div'}, style: {border: '1px solid windowframe', margin: '3px', padding: '10px', backgroundColor: 'window', display: 'none'}, children: [
           {classname: 'jls/html/Table', attributes: {fixedSize: 2}, style: {display: 'inline-table'}, children: [
              {classname: 'jls/gui/Label', attributes: {text: '#user.name:'}},
              {classname: 'jls/gui/Edit', attributes: {id: 'userId'}},
              {classname: 'jls/gui/Label', attributes: {text: '#user.password:'}},
              {classname: 'jls/gui/Edit', attributes: {id: 'password', type: 'password'}}
           ]},
           {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
           {classname: 'jls/gui/Button', attributes: {id: 'ok', text: '#ok'}, events: {click: 'onConnectOk'}},
           {classname: 'jls/gui/Button', attributes: {id: 'cancel', text: '#cancel'}, events: {click: 'onCloseMenu'}},
           {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
           {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: 'mailto:' + this.adminEMail, target: '_blank', textContent: '#admin.contact'}, style: {display: 'block'}},
           {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#user.create'}, style: {display: 'block'}, events: {click: 'onRegister'}},
           {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#user.lostPassword'}, style: {display: 'block'}, events: {click: 'onLostPassword'}}
           ]}, this.header);
        this.options = new HtmlElement({attributes: {htmlTagName: 'div'}, style: {border: '1px solid windowframe', margin: '3px', padding: '10px', backgroundColor: 'window', display: 'none'}, children: [
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#logoff'}, style: {display: 'block'}, events: {click: 'onLogoff'}},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#user.manageAccount'}, style: {display: 'block'}, events: {click: 'onManageAccount'}},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: 'help.' + _native.core.properties['user.language'] + '.html', target: '_blank', textContent: '#help'}, style: {display: 'block'}},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: 'private/misc/', target: '_blank', textContent: '#misc'}, style: {display: 'block'}},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: 'mailto:' + this.adminEMail, target: '_blank', textContent: '#admin.contact'}, style: {display: 'block'}},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div', id: 'adminOptions'}, style: {display: 'none'}, children: [
                 {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#user.list'}, style: {display: 'block'}, events: {click: 'onDisplayUsers'}},
                 {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'a', href: '#', textContent: '#user.feed'}, style: {display: 'block'}, events: {click: 'onFeedUsers'}}
             ]},
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
             {classname: 'jls/gui/Button', attributes: {id: 'cancel', text: '#cancel'}, events: {click: 'onCloseMenu'}}
         ]}, this.header);
        // fetch elements
        this.borderedFix = this.viewport.getById('bordered_fix');
        this.borderedAuto = this.viewport.getById('bordered_auto');
        this.borderedFixRow = this.viewport.getById('bordered_fix_row');
        this.borderedAutoRow = this.viewport.getById('bordered_auto_row');
        this.mediaIcon = this.toolbar.getById('mediaIcon');
        this.boardIcon = this.toolbar.getById('boardIcon');
        this.groupIcon = this.toolbar.getById('groupIcon');
        this.mediaToolbar = this.toolbar.getById('mediaToolbar');
        this.connectLink = this.header.getById('connectLink');
    },
    formatDate: function(date) {
        return Formatter.format(this.dateFormat, date);
    },
    onOpenMenu: function(event) {
        Logger.getInstance().trace('onOpenMenu');
        event.message.preventDefault();
        if (this.menu != null) {
            this.onCloseMenu();
            return;
        }
        if (this.connected) {
            this.menu = this.options;
        } else {
            if (this.userId) {
                this.logon.getById('userId').setText(this.userId);
                this.logon.getById('password').setText('');
            }
            this.menu = this.logon;
        }
        this.menu.getStyle().setProperty('display', 'block');
    },
    onCloseMenu: function(event) {
        this.menu.getStyle().setProperty('display', 'none');
        this.menu = null;
   },
    onLogoff: function(event) {
        event.message.preventDefault();
        Logger.getInstance().trace('logoff');
        this.userId = '';
        this.password = '';
        this.user = null;
        this.rememberMe = false;
        this.connected = false;
        this.connectLink.setTextContent('#logon');
        Cookie.clear('userId');
        Cookie.clear('password');
        this.onCloseMenu();
    },
    onConnectOk: function(event) {
        var userIdEdit = this.logon.getById('userId');
        var passwordEdit = this.logon.getById('password');
        this.userId = userIdEdit.getText();
        this.password = Launcher.sha1(passwordEdit.getText());
        this.rememberMe = true; // TODO use a check box
        this.onCloseMenu();
        this.broker.publish('onLogon');
    },
    onRegister: function(event) {
        Logger.getInstance().trace('onRegister');
        event.message.preventDefault();
        var div = new MessageBox({children: [
                 {classname: 'jls/html/Table', attributes: {fixedSize: 2}, style: {display: 'inline-table'}, children: [
                      {classname: 'jls/gui/Label', attributes: {text: '#user.name:'}},
                      {classname: 'jls/gui/Edit', attributes: {id: 'userId'}},
                      {classname: 'jls/gui/Label', attributes: {text: '#user.password:'}},
                      {classname: 'jls/gui/Edit', attributes: {id: 'password', type: 'password'}},
                      {classname: 'jls/gui/Label', attributes: {text: '#user.email:'}},
                      {classname: 'jls/gui/Edit', attributes: {id: 'email'}}
                  ]},
                  {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
                  {classname: 'jls/gui/Button', attributes: {text: '#ok', id: 'ok'}},
                  {classname: 'jls/gui/Button', attributes: {text: '#cancel', id: 'cancel'}}
                                             ]}, this.parent);
        div.getById('ok').observe('click', (function (e) {
            var userId = div.getById('userId').getText();
            var password = Launcher.sha1(div.getById('password').getText());
            var email = div.getById('email').getText();
            var envelope = new SxeEnvelope();
            envelope.addRequest({name: 'createUser', attributes: {login: userId, password: password, email: email}});
            this.sxe.send(envelope, (function(response) {
                var envelope = response.getEnvelope();
                Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
                var result = envelope.getResult(0);
                if (result.getName() == 'success') {
                    Logger.getInstance().trace('User created');
                    div.destroy();
                    this.broker.publish('onDisplayInfo', this.resource.get('user.create.success'));
                } else {
                    var msg = this.resource.get('user.create.failure');
                    if (result.getName() == 'failure') {
                        msg = this.resource.getf('error.message', result.getText());
                    }
                    this.broker.publish('onDisplayInfo', msg);
                }
            }).bind(this));
        }).bind(this));
        div.getById('cancel').observe('click', function (e) {
            div.destroy();
        });
    },
    onLostPassword: function(event) {
        Logger.getInstance().trace('onLostPassword()');
        event.message.preventDefault();
        var div = new MessageBox({children: [
                 {classname: 'jls/html/Table', attributes: {fixedSize: 2}, style: {display: 'inline-table'}, children: [
                      {classname: 'jls/gui/Label', attributes: {text: '#user.password.new:'}},
                      {classname: 'jls/gui/Edit', attributes: {id: 'password', type: 'password'}},
                      {classname: 'jls/gui/Label', attributes: {text: '#user.email:'}},
                      {classname: 'jls/gui/Edit', attributes: {id: 'email'}}
                  ]},
                  {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
                  {classname: 'jls/gui/Button', attributes: {text: '#ok', id: 'ok'}},
                  {classname: 'jls/gui/Button', attributes: {text: '#cancel', id: 'cancel'}}
                                             ]}, this.parent);
        div.getById('ok').observe('click', (function (e) {
            var password = Launcher.sha1(div.getById('password').getText());
            var email = div.getById('email').getText();
            var envelope = new SxeEnvelope();
            envelope.addRequest({name: 'lostUserPassword', attributes: {email: email, password: password}});
            this.sxe.send(envelope, (function(response) {
                var envelope = response.getEnvelope();
                Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
                var result = envelope.getResult(0);
                if (result.getName() == 'success') {
                    Logger.getInstance().trace('lostUserPassword succeed');
                    div.destroy();
                    this.broker.publish('onDisplayInfo', this.resource.get('user.lostPassword.success'));
                } else {
                    var msg = this.resource.get('error');
                    if (result.getName() == 'failure') {
                        msg = this.resource.getf('error.message', result.getText());
                    }
                    this.broker.publish('onDisplayInfo', msg);
                }
            }).bind(this));
        }).bind(this));
        div.getById('cancel').observe('click', function (e) {
            div.destroy();
        });
    },
    onManageAccount: function(event) {
        Logger.getInstance().trace('onManageAccount()');
        event.message.preventDefault();
        this.onCloseMenu();
        var user = this.user;
        var div = new MessageBox({children: [
             {classname: 'jls/html/Table', attributes: {fixedSize: 2}, style: {display: 'inline-table'}, children: [
                    {classname: 'jls/gui/Label', attributes: {text: '#user.name:'}},
                    {classname: 'jls/gui/Edit', attributes: {id: 'login', text: user.login}},
                    {classname: 'jls/gui/Label', attributes: {text: '#user.password.old:'}},
                    {classname: 'jls/gui/Edit', attributes: {id: 'oldPassword', type: 'password'}},
                    {classname: 'jls/gui/Label', attributes: {text: '#user.password.new:'}},
                    {classname: 'jls/gui/Edit', attributes: {id: 'newPassword', type: 'password'}},
                    {classname: 'jls/gui/Label', attributes: {text: '#user.email:'}},
                    {classname: 'jls/gui/Edit', attributes: {id: 'email', text: user.email}},
                    {classname: 'jls/gui/Label', attributes: {text: '#user.alert:'}},
                    {classname: 'jls/gui/ComboBox', attributes: {id: 'cbAlert'}}
                ]},
                {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'br'}},
                {classname: 'jls/gui/Button', attributes: {text: '#modify', id: 'modify'}},
                {classname: 'jls/gui/Button', attributes: {text: '#cancel', id: 'cancel'}}
         ]}, this.parent);
        
        var cbAlert = div.getById('cbAlert');
        var alertMat = [0, 1, 16];
        cbAlert.addString(this.resource.get('user.alert.none'));
        var currentIndex = user.alertThreshold;
        for (var i = 1; i < alertMat.length; i++) {
            cbAlert.addString(this.resource.getf(i > 1 ? 'user.alert.counts' : 'user.alert.count', alertMat[i]));
            if (user.alertThreshold == alertMat[i]) {
                currentIndex = i;
            }
        }
        cbAlert.setCurrentSelectionIndex(currentIndex);
        div.getById('modify').observe('click', (function (e) {
            var login = div.getById('login').getText();
            var oldPassword = div.getById('oldPassword').getText();
            if (oldPassword) {
                oldPassword = Launcher.sha1(oldPassword);
            }
            var newPassword = div.getById('newPassword').getText();
            if (newPassword) {
                newPassword = Launcher.sha1(newPassword);
            }
            var email = div.getById('email').getText();
            var alertThreshold = cbAlert.getCurrentSelectionIndex();
            alertThreshold = alertMat[alertThreshold];
            var modifyUserAttr = {};
            if ((login) && (login != user.login)) {
                modifyUserAttr.login = login;
            }
            if ((oldPassword) && (newPassword) && (newPassword != user.password)) {
                modifyUserAttr.password = newPassword;
            }
            if ((email) && (email != user.email)) {
                modifyUserAttr.email = email;
            }
            if (alertThreshold != user.alertThreshold) {
                modifyUserAttr.alertThreshold = alertThreshold;
            }
            var envelope = new SxeEnvelope(this.userId, oldPassword || this.password);
            envelope.addRequest({name: 'modifyUser', attributes: modifyUserAttr});
            Logger.getInstance().trace('request: -->' + envelope.toString() + '<--');
            this.sxe.send(envelope, (function(response) {
                var envelope = response.getEnvelope();
                Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
                var result = envelope.getResult(0);
                if (result.getName() == 'ModifyUserResult') {
                    // update user
                    this.user = User.createFromXML(result.getChild(0));
                    Logger.getInstance().trace('User modified');
                    div.destroy();
                    this.broker.publish('onDisplayInfo', this.resource.get('user.modify.success'));
                } else {
                    var msg = this.resource.get('error');
                    if (result.getName() == 'failure') {
                        msg = this.resource.getf('error.message', result.getText());
                    }
                    this.broker.publish('onDisplayInfo', msg);
                }
            }).bind(this));
        }).bind(this));
        div.getById('cancel').observe('click', function (e) {
            div.destroy();
        });
    },
    displayInfo: function(element) {
        Logger.getInstance().trace('displayInfo()');
        var div = new MessageBox({children: [
                {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div'}, style: {margin: '30px'}, children: [
                    {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div'}, children: [element]},
                    {classname: 'jls/gui/Button', attributes: {text: '#ok', id: 'ok'}}
                ]}
         ]}, this.parent);
        div.getById('ok').observe('click', function () {
            div.destroy();
        });
    },
    onDisplayInfo: function(event) {
        Logger.getInstance().trace('onDisplayInfo');
        this.displayInfo({classname: 'jls/gui/Label', attributes: {text: event.message}});
    },
    onFeedUsers: function(event) {
        Logger.getInstance().trace('onFeedUsers');
        event.message.preventDefault();
        var div = new MessageBox({children: [
             {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div'}, children: [
                  {classname: 'jls/gui/Label', attributes: {text: '#user.feed.confirm'}}
             ]},
             {classname: 'jls/gui/Button', attributes: {text: '#ok', id: 'ok'}},
             {classname: 'jls/gui/Button', attributes: {text: '#cancel', id: 'cancel'}}
             ]}, this.parent);
        div.getById('ok').observe('click', (function () {
            div.destroy();
            this.onFeedUsersConfirmed();
        }).bind(this));
        div.getById('cancel').observe('click', function () {
            div.destroy();
        });
    },
    onFeedUsersConfirmed: function() {
        Logger.getInstance().trace('onFeedUsersConfirmed');
        var envelope = new SxeEnvelope(this.userId, this.password);
        envelope.addRequest({name: 'feedUser'});
        this.sxe.send(envelope, (function(response) {
            var envelope = response.getEnvelope();
            Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
            var result = envelope.getResult(0);
            var msg;
            if (result.getName() == 'userFeedResult') {
                var emailCount = parseInt(result.getAttribute('emailCount'), 10);
                var failureCount = parseInt(result.getAttribute('failureCount'), 10);
                msg = this.resource.getf('user.feed.result', emailCount, failureCount);
            } else {
                var msg = this.resource.get('error');
                if (result.getName() == 'failure') {
                    msg = this.resource.getf('error.message', result.getText());
                }
            }
            this.broker.publish('onDisplayInfo', msg);
        }).bind(this));
    },
    onDisplayUsers: function(event) {
        event.message.preventDefault();
        this.onCloseMenu();
        var users = [];
        for (var id in this.users) {
            users.push(this.users[id]);
        }
        users.sort(function(a, b) {
                if (a.lastAuthenticationDate < b.lastAuthenticationDate)
                    return 1;
                if (a.lastAuthenticationDate > b.lastAuthenticationDate)
                    return -1;
                return 0;
            });
        Logger.getInstance().trace('onDisplayUsers ' + users.length + ' user(s)');
        var element = {classname: 'jls/html/Table', attributes: {fixedSize: 5, cellStyle: {padding: '5px', border: '1px dotted black'}}, style: {borderCollapse: 'collapse', display: 'inline-table'}, children: [
            {classname: 'jls/gui/Label', attributes: {text: '#user.id'}},
            {classname: 'jls/gui/Label', attributes: {text: '#user.name'}},
            {classname: 'jls/gui/Label', attributes: {text: '#user.email'}},
            {classname: 'jls/gui/Label', attributes: {text: '#user.creationDate'}},
            {classname: 'jls/gui/Label', attributes: {text: '#user.lastLogonDate'}}
        ]};
        for (var i = 0; i < users.length; i++) {
            var user = users[i];
            element.children.push({classname: 'jls/gui/Label', attributes: {text: user.id}});
            element.children.push({classname: 'jls/gui/Label', attributes: {text: user.login}});
            element.children.push({classname: 'jls/gui/Label', attributes: {text: user.email}});
            element.children.push({classname: 'jls/gui/Label', attributes: {text: this.formatDate(user.creationDate)}});
            element.children.push({classname: 'jls/gui/Label', attributes: {text: this.formatDate(user.lastAuthenticationDate)}});
        }
        this.displayInfo(element);
    },
    onLogon: function() {
        Logger.getInstance().trace('onLogon');
        var envelope = new SxeEnvelope();
        envelope.setAuthentication(this.userId, this.password);
        envelope.addRequest({name: 'authenticateUser'});
        envelope.addRequest({name: 'listUsers'});
        this.sxe.send(envelope, this.broker.callback('onAuthenticated'));
    },
    onAuthenticated: function(event) {
        Logger.getInstance().trace('onAuthenticated');
        var envelope = event.message.getEnvelope();
        Logger.getInstance().debug('envelope: -->' + envelope.toString() + '<--');
        var result = envelope.getResult(0);
        if (result.getName() != 'AuthenticateUserResult') {
            Logger.getInstance().trace('Authentication fails');
            var msg = this.resource.get('logon.failure');
            if (result.getName() == 'failure') {
                msg = this.resource.getf('error.message', result.getText());
            }
            this.broker.publish('onDisplayInfo', msg);
            return;
        }
        Logger.getInstance().trace('AuthenticateUser succeed');
        if (this.rememberMe) {
            Logger.getInstance().trace('Place credentials in cookie');
            var nextYear = new Date();
            nextYear.setFullYear(nextYear.getFullYear() + 1);
            Cookie.set('userId', this.userId, {expires: nextYear});
            Cookie.set('password', this.password, {expires: nextYear});
        }
        this.connected = true;
        this.connectLink.setTextContent(this.userId);
        Logger.getInstance().trace('connected');

        this.users = {};
        var userListResult = envelope.getResult(1);
        //Logger.getInstance().trace('Users: --> ' + userListResult.toString('UTF-8') + '<--');
        if (userListResult.getName() == 'userListResult') {
            for (var i = 0; i < userListResult.getChildCount(); i++) {
                var user = User.createFromXML(userListResult.getChild(i));
                this.users[user.id] = user;
                Logger.getInstance().trace('User id ' + user.id + ' added');
            }
        }

        this.user = User.createFromXML(result.getChild(0));
        if (! ('read' in this.user.rights)) {
            this.broker.publish('onDisplayInfo', this.resource.get('logon.badRights'));
        }
        this.borderedAuto.removeChildren();
        new HtmlElement({attributes: {htmlTagName: 'img', src: 'style/progress.gif'}}, this.borderedAuto);
        new XMLHttpRequest('private/access.php').setAuthentication(this.userId, this.password).setMethod('post').send(null, this.broker.callback('onAccessGranted'));
        this.options.getById('adminOptions').getStyle().setProperty('display', 'admin' in this.user.rights ? '' : 'none');
    },
    onAccessGranted: function() {
        Logger.getInstance().trace('onAccessGranted');
        var envelope = new SxeEnvelope();
        envelope.setAuthentication(this.userId, this.password);
        envelope.addRequest({name: 'listMedias'});
        envelope.addRequest({name: 'listCommentedMedias', attributes: {limit: 40}});
        this.sxe.send(envelope, this.broker.callback('onMediasLoaded'));
    },
    computeRootView : function(medias, commentedMedias, lastAuthenticationDate) {
        Logger.getInstance().info('computeRootView() ' + medias.length + ' media(s)');
        var root = new Group.View('-');
        if (medias.length == 0) {
            return root;
        }
        medias.sort(Media.compareMedias);
        // new medias
        if (lastAuthenticationDate != 0) {
            var mediasByPubDate = medias.concat([]);
            mediasByPubDate.sort(Media.compareMediasByPubDate);
            var recents = new Group.Medias(this.resource.get('board.recent'));
            var last = Math.max(0, mediasByPubDate.length - Launcher.MAX_MEDIA * 4);
            for (var i = mediasByPubDate.length - 1; (i >= last) && (mediasByPubDate[i]._pubDate > lastAuthenticationDate); i--) {
                recents.addItem(mediasByPubDate[i]);
            }
            recents.sort(Media.compareMedias);
            if (recents.size() > 0) {
                root.addItem(recents);
            }
        }
        // by date
        var rootByDate = new Group.List(this.resource.get('board.byDate'));
        var year = -1;
        var yearGroup = null;
        var monthFullNames = Formatter.getResource(_native.core.properties['user.language']).monthFullNames;
        var month = -1;
        var monthGroup = null;
        for (var i = 0; i < medias.length; i++) {
            var media = medias[i];
            var yearMedia = media.getDate().getFullYear();
            if (year != yearMedia) {
                yearGroup = new Group.List(yearMedia);
                year = yearMedia;
                rootByDate.addItem(yearGroup);
                month = -1;
                monthGroup = null;
            }
            var monthMedia = media.getDate().getMonth();
            if (month != monthMedia) {
                // create group
                monthGroup = new Group.Medias(monthFullNames[monthMedia]);
                month = monthMedia;
                yearGroup.addItem(monthGroup);
            }
            monthGroup.addItem(media);
        }
        root.addItem(rootByDate);
        // Commented medias
        var commentMap = {};
        if (commentedMedias) {
            for (var i = 0; i < commentedMedias.length; i++) {
                var cm = commentedMedias[i];
                commentMap[cm.mediaId] = cm.lastDate;
            }
        }
        // Films
        var movies = new Group.Medias(this.resource.get('board.movies'));
        var comments = new Group.RevertedMedias(this.resource.get('board.commented'));
        for (var i = 0; i < medias.length; i++) {
            var m = medias[i];
            if (m instanceof Media.Movie) {
                movies.addItem(m);
            }
            if (commentMap[m.getId()]) {
                comments.addItem(m);
            }
        }
        if (movies.size() > 0) {
            root.addItem(movies);
        }
        if (comments.size() > 0) {
            Logger.getInstance().info('sort comments, first: ' + comments.getItem(0).getId() + ', date: ' + commentMap[comments.getItem(0).getId()]);
            comments.sort(function(a, b) {
                var aDate = commentMap[a.getId()];
                var bDate = commentMap[b.getId()];
                if (aDate < bDate)
                    return -1;
                if (aDate > bDate)
                    return 1;
                return 0;
            });
            Logger.getInstance().info('sorted comments, first: ' + comments.getItem(0).getId() + ', date: ' + commentMap[comments.getItem(0).getId()]);
            root.addItem(comments);
        }
        // Random
        var rand = new Group.Medias(this.resource.get('board.random'));
        for (var i = 0; i < Launcher.MAX_MEDIA; i++) {
            var m = medias[Math.floor(Math.random() * medias.length)];
            rand.addItem(m);
        }
        root.addItem(rand);
        return root;
    },
    onMediasLoaded: function(event) {
        Logger.getInstance().trace('onMediasLoaded()');
        var envelope = event.message.getEnvelope();
        var result = envelope.getResult(0);
        if (result.getName() != 'mediaListResult') {
            return;
        }
        var mediaNodes = result.getChildren();
        this.medias = [];
        for (var i = 0; i < mediaNodes.length; i++) {
            this.medias.push(Media.createFromXML(mediaNodes[i]));
        }
        result = envelope.getResult(1);
        if (result.getName() != 'commentListResult') {
            return;
        }
        var commentedMedias = [];
        var commentNodes = result.getChildren();
        for (var i = 0; i < commentNodes.length; i++) {
            var node = commentNodes[i];
            var mediaId = parseInt(node.getAttribute('mediaId'), 10);
            var lastDate = new Date(parseInt(node.getAttribute('lastDate'), 10) * 1000);
            commentedMedias.push({
                mediaId: mediaId,
                lastDate: lastDate
            });
        }
        this.root = this.computeRootView(this.medias, commentedMedias, this.user.lastAuthenticationDate);
        this.changeContext(new Context('group', this.root, 0));
        this.toolbar.getStyle().setProperty('display', '');
    },
    onDisplayMedia: function(event) {
        event.message.preventDefault();
        this.changeContext(this.currentContext.createView('media'));
    },
    onDisplayBoard: function(event) {
        event.message.preventDefault();
        this.changeContext(this.currentContext.createView('board'));
    },
    onDisplayBoardFull: function(event) {
        event.message.preventDefault();
        // TODO Add a sticky tag full on the group and just display the board
        this.changeContext(this.currentContext.createView('boardFull'));
    },
    onDisplayGroup: function(event) {
        event.message.preventDefault();
        var group = this.currentContext.group;
        if (group instanceof Group.Medias) {
            group = group.getParent();
        }
        this.changeContext(new Context('group', group, 0));
    },
    displayMedia: function(context) {
        Logger.getInstance().trace('displayMedia()');
        var media = context.getMedia();
        this.borderedFix.removeChildren();
        this.borderedAuto.removeChildren();
        media.createElement({group: context.group, index: context.mediaIndex, attributes: {'class': 'media'}}, this.borderedFix);
        this.borderedFixRow.getStyle().setProperty('display', '');
        this.borderedAutoRow.getStyle().setProperty('display', 'none');
        this.mediaToolbar.getStyle().setProperty('display', '');
        
        if (media.getCommentCount() == 0) {
            return;
        }
        this.borderedAutoRow.getStyle().setProperty('display', '');
        if (media.getCommentCount() != media.getComments().length) {
            this.borderedAuto.removeChildren();
            new HtmlElement({attributes: {htmlTagName: 'img', src: 'style/progress.gif'}}, this.borderedAuto);
            var envelope = new SxeEnvelope(this.userId, this.password);
            envelope.addRequest({name: 'listComment', attributes: {mediaId: media.getId()}});
            this.sxe.send(envelope, (function(response) {
                var envelope = response.getEnvelope();
                Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
                var result = envelope.getResult(0);
                if (result.getName() == 'commentListResult') {
                    for (var i = 0; i < result.getChildCount(); i++) {
                        var comment = Comment.createFromXML(result.getChild(i));
                        media.addComment(comment);
                        Logger.getInstance().trace('Comment id ' + comment.getId() + ' added');
                    }
                }
                this.displayMediaComments(context);
            }).bind(this));
        } else {
            this.displayMediaComments(context);
        }
    },
    displayMediaComments: function(context) {
        this.borderedAuto.removeChildren();
        var media = context.getMedia();
        for (var i = 0; i < media.getComments().length; i++) {
            var comment = media.getComments()[i];
            var commentElements = [
                {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div', 'class': 'commentHeader',
                    textContent: this.formatDate(comment.getDate()) + ', ' + this.users[comment.getUserId()].login}}
            ];
            var lines = comment.getContent().split('\n');
            for (var j = 0; j < lines.length; j++) {
                commentElements.push({classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'div', textContent: lines[j]}});
            }
            new HtmlElement({attributes: {htmlTagName: 'div', 'class': 'comment'}, children: commentElements}, this.borderedAuto);
        }
    },
    createBreadcrumb: function(group, parent) {
        var groups = [];
        while (group) {
            groups.push(group);
            group = group.getParent();
        }
        if (groups.length == 1) {
            return;
        }
        var p = new HtmlElement({attributes: {htmlTagName: 'p'}, style: {textAlign: 'left'}}, parent);
        for (var i = groups.length - 1; i >= 0; i--) {
            if (i > 0) {
                var e = new HtmlElement({attributes: {htmlTagName: 'a', href: '#', textContent: groups[i].getName()}, events: {click: 'onOpenGroup'}}, p);
                e._group = groups[i];
                e.getGroup = GroupPreview.prototype.getGroup;
                new HtmlElement({attributes: {htmlTagName: 'span', htmlContent: '&nbsp;&raquo;&nbsp;'}}, p);
            } else {
                new HtmlElement({attributes: {htmlTagName: 'span', textContent: groups[i].getName()}}, p);
            }
        }
    },
    displayBoard: function(context, full) {
        Logger.getInstance().trace('displayBoard()');
        this.borderedAuto.removeChildren();
        this.createBreadcrumb(context.group, this.borderedAuto);
        var table = new Table({attributes: {'class': 'board', fixedSize: 4, cellAttributes: {'class': 'preview'}}}, this.borderedAuto);
        var mediaCount = context.group.getMediaCount();
        if ((! (full || (context.group instanceof Group.Medias))) && (mediaCount > Launcher.MAX_MEDIA)) {
            //new HtmlElement({attributes: {htmlTagName: 'img', src: 'style/list-add.png'}, style: {margin: '30px', cursor: 'pointer'}, events: {click: 'onDisplayBoardFull'}}, this.borderedAuto);
            new HtmlElement({attributes: {htmlTagName: 'a', href: '#', textContent: this.resource.getf('seeAllMedias', mediaCount)},
                style: {margin: '30px'}, events: {click: 'onDisplayBoardFull'}}, this.borderedAuto);
            mediaCount = Launcher.MAX_MEDIA;
        }
        for (var i = 0; i < mediaCount; i++) {
            var media = context.group.getMedia(i);
            Logger.getInstance().trace('createMiniature(' + i + ')');
            media.createMiniature({group: context.group, index: i, attributes: {'class': 'media'}, events: {click: 'onOpenMedia'}}, table);
        }
        this.borderedFixRow.getStyle().setProperty('display', 'none');
        this.borderedAutoRow.getStyle().setProperty('display', '');
        this.mediaToolbar.getStyle().setProperty('display', 'none');
    },
    displayGroup: function(context) {
        Logger.getInstance().trace('displayGroup()');
        this.borderedAuto.removeChildren();
        this.createBreadcrumb(context.group, this.borderedAuto);
        var table = new Table({attributes: {'class': 'board', fixedSize: 4}}, this.borderedAuto);
        var size = context.group.size();
        for (var i = 0; i < size; i++) {
            var item = context.group.getItem(i);
            new GroupPreview({group: item, attributes: {'class': 'titled_preview'}, events: {click: 'onOpenGroup'}}, table);
        }
        this.borderedFixRow.getStyle().setProperty('display', 'none');
        this.borderedAutoRow.getStyle().setProperty('display', '');
        //this.borderedAuto.getStyle().removeProperty('display');
        this.mediaToolbar.getStyle().setProperty('display', 'none');
    },
    displayContext: function(context) {
        Logger.getInstance().trace('displayContext(' + context.viewName + ')');
        this.mediaIcon.setAttribute('class', 'icon');
        this.boardIcon.setAttribute('class', 'icon');
        this.groupIcon.setAttribute('class', 'icon');
        switch (context.viewName) {
        case 'media':
            this.mediaIcon.setAttribute('class', 'iconSelected');
            this.displayMedia(context);
            break;
        case 'board':
        case 'boardFull':
            this.boardIcon.setAttribute('class', 'iconSelected');
            this.displayBoard(context, context.viewName == 'boardFull');
            break;
        case 'group':
            this.groupIcon.setAttribute('class', 'iconSelected');
            this.displayGroup(context);
            break;
        }
    },
    changeContext: function(context) {
        Logger.getInstance().trace('changeContext(' + context.viewName + ')');
        var onlyMedia = false;
        if ((this.currentContext != null) &&
            (this.currentContext.viewName == context.viewName) &&
            (this.currentContext.group == context.group))
        {
            if (this.currentContext.mediaIndex == context.mediaIndex) {
                return;
            } else {
                onlyMedia = true;
            }
        }
        // Reset navigation context
        this.comment = null;
        
        // Save scroll position
        var scrollPos = this.viewport._htmlElement.scrollTop;
        if (this.currentContext != null) {
            switch (this.currentContext.viewName) {
            case 'board':
                this.currentContext.group._scrollBoardHistory = scrollPos;
                break;
            case 'group':
                this.currentContext.group._scrollGroupHistory = scrollPos;
                break;
            }
        }
        Logger.getInstance().trace('Saving scroll position: ' + scrollPos);
        
        this.currentContext = context;
        
        // History
        if (this.contextHistoryIndex != this.contextHistory.length - 1) {
            this.contextHistory.length = this.contextHistoryIndex + 1;
        }
        this.contextHistoryIndex++;
        this.contextHistory[this.contextHistoryIndex] = this.currentContext;
        if (onlyMedia) {
            this.displayMedia(this.currentContext);
        } else {
            this.displayContext(this.currentContext);
        }

        // Restore scroll position
        var scrollPos = -1;
        switch (this.currentContext.viewName) {
        case 'board':
            scrollPos = 0;
            if ('_scrollBoardHistory' in this.currentContext.group) {
                scrollPos = this.currentContext.group._scrollBoardHistory;
            }
            break;
        case 'group':
            scrollPos = 0;
            if ('_scrollGroupHistory' in this.currentContext.group) {
                scrollPos = this.currentContext.group._scrollGroupHistory;
            }
            break;
        }
        if (scrollPos >= 0) {
            Logger.getInstance().trace('Restore scroll position: ' + scrollPos);
            this.viewport._htmlElement.scrollTop = scrollPos;
        }
    },
    onGoNext: function(event) {
        event.message.preventDefault();
        if (this.contextHistoryIndex < this.contextHistory.length - 1) {
            this.contextHistoryIndex++;
            this.currentContext = this.contextHistory[this.contextHistoryIndex];
            this.displayContext(this.currentContext);
        } else {
            alert('no more history ' + this.contextHistoryIndex + '/' + this.contextHistory.length);
        }
    },
    onGoPrevious: function(event) {
        event.message.preventDefault();
        if (this.contextHistoryIndex > 0) {
            this.contextHistoryIndex--;
            this.currentContext = this.contextHistory[this.contextHistoryIndex];
            this.displayContext(this.currentContext);
        } else {
            alert('no more history ' + this.contextHistoryIndex + '/' + this.contextHistory.length);
        }
    },
    onGoHome: function(event) {
        event.message.preventDefault();
        this.changeContext(new Context('group', this.root, 0));
    },
    openGroup: function(grp) {
        this.changeContext(new Context((grp instanceof Group.Medias) ? 'board' : 'group', grp, 0));
    },
    onOpenGroup: function(event) {
        event.message.preventDefault();
        this.openGroup(event.message.element.getGroup());
    },
    onOpenMedia: function(event) {
        event.message.preventDefault();
        var grp = event.message.element.getGroup();
        var idx = event.message.element.getIndex();
        //var media = event.message.element.getMedia();
        this.changeContext(new Context('media', grp, idx));
    },
    onComment: function(event) {
        event.message.preventDefault();
        if (this.comment) {
            return;
        }
        this.borderedAutoRow.getStyle().setProperty('display', '');
        var div = new HtmlElement({attributes: {htmlTagName: 'div'}, style: {padding: '30px', textAlign: 'right'},
            children: [
                       {classname: 'jls/html/HtmlElement', attributes: {htmlTagName: 'textarea', id: 'newComment'}, style: {display: 'block', width: '100%'}},
                       {classname: 'jls/gui/Button', attributes: {text: '#ok', id: 'ok'}},
                       {classname: 'jls/gui/Button', attributes: {text: '#cancel', id: 'cancel'}}
            ]}, this.borderedAuto);
        div.getById('ok').observe('click', (function (e) {
            var media = this.currentContext.getMedia();
            var content = div.getById('newComment').getHtmlElement().value;
            Logger.getInstance().debug('content: -->' + content + '<--');
            var envelope = new SxeEnvelope(this.userId, this.password);
            envelope.addRequest({name: 'addComment', attributes: {mediaId: media.getId(), content: content}});
            this.sxe.send(envelope, (function(response) {
                var envelope = response.getEnvelope();
                Logger.getInstance().debug('response: -->' + envelope.toString() + '<--');
                var result = envelope.getResult(0);
                if (result.getName() == 'success') {
                    Logger.getInstance().trace('Comment added');
                    media.newComment(this.user.id, content);
                    div.destroy();
                    this.comment = null;
                    this.displayMediaComments(this.currentContext);
                    this.root.getItemByName('Avec commentaire').addItem(media);
                } else {
                    var msg = this.resource.get('error');
                    if (result.getName() == 'failure') {
                        msg = this.resource.getf('error.message', result.getText());
                    }
                    this.broker.publish('onDisplayInfo', msg);
                }
            }).bind(this));
        }).bind(this));
        div.getById('cancel').observe('click', (function (e) {
            div.destroy();
            this.comment = null;
        }).bind(this));
        this.comment = true;
        div.getById('newComment').getHtmlElement().focus();
    },
    onGoBackward: function(event) {
        event.message.preventDefault();
        this.changeContext(this.currentContext.createMedia(-1));
    },
    onGoUp: function(event) {
        event.message.preventDefault();
        this.openGroup(this.currentContext.group);
    },
    onGoForward: function(event) {
        event.message.preventDefault();
        this.changeContext(this.currentContext.createMedia(1));
    }
});

Object.extend(Launcher,
{
    _instance: null,
    MAX_MEDIA: 16,
    scrollHistory : {},
    sha1: function(value) {
        var md = MessageDigest.getInstance('SHA1');
        md.updateString(value, 'UTF-8');
        var buffer = md.digest();
        return StringCodec.hexEncode(buffer);
    },
    getInstance: function() {
        if (Launcher._instance == null) {
            Launcher._instance = new Launcher();
        }
        return Launcher._instance;
    }
});


return Launcher;
});
