/*
 * Asynchronous Module Definition (AMD) API implementation for JLS.
 * The following code creates a 'require' function to load modules.
 */
(function() {
    /* Miscellaneous functions */
    var warn, log, debug, basename, dirname, concatPath, evalScript;
    log = debug = warn = function(msg) {
        //_native.core.log(4, msg);
    };
    evalScript = function(path, callback, sync, prefix, suffix) {};
    basename = function(path) {
        var index = path.lastIndexOf('/');
        return index == -1 ? path : path.substring(index + 1);
    };
    dirname = function(path) {
        var index = path.lastIndexOf('/');
        return index == -1 ? '' : path.substring(0, index);
    };
    concatPath = function(a, b) {
        var pathItems = b.split('/');
        if ((a != '.') && ((pathItems[0] == '.') || (pathItems[0] == '..'))) {
            pathItems = a.split('/').concat(pathItems);
        }
        for (var i = 0; i < pathItems.length; i++) {
            var pathItem = pathItems[i];
            if ((pathItem == '.') || (pathItem == '')) {
                pathItems.splice(i, 1);
                i--;
            } else if ((i > 0) && (pathItem == '..')) {
                pathItems.splice(i - 1, 2);
                i -= 2;
            }
        }
        return pathItems.join('/');
    };
    var fullyQualifiedRequire, createRequire;
    /* The cache is a map containing the defined module by absolute path (module identifier) */
    var cache = {};
    /* The AMD specific define entry */
    var amd = {
            multiversion: true
    };
    /*  */
    var CacheEntry = function(path, sync, module) {
        this.path = path || null;
        this.module = module || undefined;
        this.sync = sync || false;
        this.paths = null;
        this.callback = null;
        this.dependencies = [];
        this.count = 0;
        this.waiters = [];
        if (path) {
            cache[path] = this;
        }
        log('CacheEntry("' + path + '", ' + sync + ', ' + module + ')');
    };
    CacheEntry.prototype.declare = function(paths, callback) {
        log('declare([' + paths + '], ...) "' + this.path + '"');
        this.paths = paths || null;
        this.callback = callback || null;
    };
    CacheEntry.prototype.onDefined = function(module) {
        log('onDefined(...) "' + this.path + '"');
        this.module = module || null;
        // Play dependencies
        for (var index = 0; index < this.waiters.length; index++) {
            this.waiters[index].notify(this);
        }
    };
    /* Creates a define function suitable for this module */
    CacheEntry.prototype.createDefine = function() {
        debug('createDefine() "' + this.path + '"');
        var self = this;
        var def = function(id, dependencies, factory) {
            // Parse arguments
            switch (arguments.length) {
            case 0:
                return;
            case 1:
                factory = id;
                dependencies = null;
                id = undefined;
                break;
            case 2:
                factory = dependencies;
                dependencies = id;
                id = undefined;
                break;
            case 3:
            default:
                break;
            }
            // TODO what do we do with the id?
            fullyQualifiedRequire(self.path, dependencies, factory, self.sync);
        };
        def.amd = amd;
        return def;
    };
    CacheEntry.prototype.onLoaded = function(fn, err) {
        debug('onLoaded(..., "' + err + '") "' + this.path + '"');
        if (err) {
            warn('Exception raised while loading "' + this.path + '": ' + err);
        } else if (typeof fn == 'function') {
            try {
                fn(this.createDefine());
                return;
            }
            catch (e) {
                warn('Exception raised while evaluating "' + this.path + '": ' + e);
            }
        } else {
            warn('Invalid argument while loading "' + this.path + '" (' + (typeof fn) + ')');
        }
        this.onDefined(null);
    };
    CacheEntry.prototype.load = function() {
        debug('load() "' + this.path + '"');
        var self = this;
        evalScript(this.path + '.js', function(fn, e) {
            self.onLoaded.call(self, fn, e);
        }, this.sync, '(function() { return function(define) { ', ' }; })();');
    };
    CacheEntry.prototype.notify = function(entry) {
        debug('notify("' + entry.path + '") "' + this.path + '"');
        for (var index = 0; index < this.paths.length; index++) {
            var path = this.paths[index];
            if (path == entry.path) {
                this.setDependency(index, entry.module);
                break;
            }
        }
    };
    CacheEntry.prototype.isDependent = function(entry) {
        debug('isDependent("' + entry.path + '") "' + this.path + '"');
        if (entry.path == null) {
            return false;
        }
        for (var index = 0; index < this.waiters.length; index++) {
            var waiter = this.waiters[index];
            if ((waiter.path != null) && ((waiter.path === entry.path) || (waiter.isDependent(entry)))) {
                return true;
            }
        }
        return false;
    };
    CacheEntry.prototype.wait = function(entry) {
        debug('wait("' + entry.path + '") "' + this.path + '"');
        if (this.isDependent(entry)) {
            throw 'Cyclic dependency detected between ' + this.path + ' and ' + entry.path;
        }
        entry.waiters.push(this);
    };
    CacheEntry.prototype.isLoaded = function() {
        return typeof this.module != 'undefined';
    };
    CacheEntry.prototype.isModule = function() {
        return this.path != null;
    };
    CacheEntry.prototype.isDeclared = function() {
        return (this.callback != null) || this.isLoaded();
    };
    CacheEntry.prototype.setDependency = function(index, value) {
        this.dependencies[index] = value;
        this.count++;
        debug('setDependency(' + index + ', ...) "' + this.path + '" ' + this.count + '/' + this.paths.length);
        this.wakeup();
    };
    CacheEntry.prototype.wakeup = function() {
        if ((this.callback) && ((this.paths == null) || (this.count == this.paths.length))) {
            debug('wakeup() "' + this.path + '"');
            var fn = this.callback;
            var dep = this.dependencies;
            this.paths = null;
            this.callback = null;
            this.dependencies = null;
            var module = fn.apply(null, dep);
            this.onDefined(this.isModule() ? module : null);
        }
    };
    /* Base require/define function that use a fully qualified module id */
    fullyQualifiedRequire = function(moduleId, ids, callback, sync) {
        if (typeof callback != 'function') {
            throw 'Invalid arguments';
        }
        sync = sync || false;
        var basepath = dirname(moduleId || '');
        var name = basename(moduleId || '');
        var paths = [];
        if ((ids != null) && ('length' in ids)) {
            for (var index = 0; index < ids.length; index++) {
                var id = ids[index];
                if (id == '.') {
                    id = './main';
                }
                paths.push(concatPath(basepath, id));
            }
        }
        if (name == '?') {
            moduleId = null;
        }
        var entry;
        if (moduleId) {
            if (moduleId in cache) {
                // Check sync ?
                entry = cache[moduleId];
            } else {
                entry = new CacheEntry(moduleId, sync, undefined);
            }
        } else {
            entry = new CacheEntry(null, sync, undefined);
        }
        entry.declare(paths, callback);
        for (var index = 0; index < paths.length; index++) {
            var path = paths[index];
            var depEntry;
            try {
                if (path == 'require') {
                    entry.setDependency(index, createRequire(basepath));
                } else if (path in cache) {
                    depEntry = cache[path];
                    if (depEntry.isLoaded()) {
                        entry.setDependency(index, depEntry.module);
                    } else {
                        // TODO Check sync
                        entry.wait(depEntry);
                    }
                } else {
                    depEntry = new CacheEntry(path, sync);
                    entry.wait(depEntry);
                    depEntry.load();
                }
            }
            catch (e) {
                warn('Exception raised: ' + e);
                //debug(e.name + ': ' + e.message);
                entry.setDependency(index, null);
            }
        }
        entry.wakeup();
    };
    // Populate default cache values
    new CacheEntry('_AMD', true, {
        cache : cache,
        setLogFunction: function(fn) {
            warn = fn;
            //log = debug = warn;
        },
        setEvalScriptFunction: function(fn) {
            evalScript = fn;
        },
        getModuleId: function(obj) {
            for (var id in cache) {
                if (cache[id].module === obj.constructor) {
                    return id;
                }
            }
            return null;
        },
        status: function() {
            log('AMD Cache status:');
            var lm = [], wm = [];
            for (var path in cache) {
                (cache[path].module ? lm : wm).push(path);
            }
            lm.sort(); wm.sort();
            for (var i = 0; i < lm.length; i++) {
                log(' "' + lm[i] + '" loaded');
            }
            for (var i = 0; i < wm.length; i++) {
                var path = wm[i];
                log(' "' + path + '" ' + cache[path].onDefine.length + ' waiting callback(s)');
            }
        },
        dirname : dirname,
        basename : basename,
        concatPath : concatPath,
        createRequire : createRequire,
        fullyQualifiedRequire : fullyQualifiedRequire
    });
    /* Returns a require function suitable for a specific path */
    createRequire = function(path) {
        var req = function(ids, callback) {
            var async = true;
            var rmod = null;
            if (typeof ids == 'string') {
                async = false;
                ids = [ids];
                callback = function(module) {
                    if ((typeof module == 'undefined') || (module == null)) {
                        warn('Fail to require "' + ids[0] + '"');
                    }
                    rmod = module;
                };
            }
            fullyQualifiedRequire(path + '/?', ids, callback, !async);
            return async ? undefined : rmod;
        };
        req.toUrl = function(s) {
            log('toUrl["' + path + '"]("' + s + '")');
            return concatPath(path, s);
        };
        return req;
    };
    // global require function
    require = createRequire('');
})();
