define(['jls/lang/Class'], function (Class) {

var Cookie;
/**
 * @class This class is a collection of Cookie helper functions.
 * @name jls.util.Cookie
 */
Cookie = Class.create();

Object.extend(Cookie, /** @lends jls.util.Cookie */
{
    /**
     * Returns a cookie depending on its name or null if there is no cookie for this name.
     * 
     * @param {String} name The name of the cookie to get.
     * @returns {String} The value or null if there is no cookie for this name.
     */
    get : function(name) {
        var cookies = document.cookie;
        var prefix = name + '=';
        var indexOfPrefix = cookies.indexOf('; ' + prefix);
        if (indexOfPrefix > 0) {
            indexOfPrefix += 2;
        } else {
            indexOfPrefix = cookies.indexOf(prefix);
            if (indexOfPrefix != 0) {
                return null;
            }
        }
        var end = document.cookie.indexOf(';', indexOfPrefix);
        if (end == -1) {
            end = cookies.length;
        }
        return unescape(cookies.substring(indexOfPrefix + prefix.length, end));
    },
    /**
     * Sets a cookie.
     * The expiration date tells the browser when to delete the cookie.
     * The domain and path tell the browser that the cookie has to be sent back to the server when requesting URLs of a given domain and path.
     * If not specified, they default to the domain and path of the object that was requested.
     * As a result, the domain and path strings may tell the browser to send the cookie when it normally would not.
     * For security reasons, the cookie is accepted only if the server is a member of the domain specified by the domain string.
     * A secure cookie is only used when a browser is visiting a server via HTTPS.
     * 
     * @param {String} name The name of the cookie to set.
     * @param {String} value The value.
     * @param {Object} [opts] The options.
     * @param {Date} [opts.expires] The expiration date.
     * @param {String} [opts.domain] The domain.
     * @param {String} [opts.path] The path.
     * @param {String} [opts.secure] The secure.
     */
    set : function(name, value, opts) {
        var cookie = name + '=' + escape(value);
        if (opts) {
            if (('expires' in opts) && (opts.expires != null)) {
                cookie += '; expires=' + opts.expires.toGMTString();
            }
            if (('domain' in opts) && (opts.domain != null)) {
                cookie += '; domain=' + opts.domain.toString();
            }
            if (('path' in opts) && (opts.path != null)) {
                cookie += '; path=' + opts.path.toString();
            }
            if (('secure' in opts) && opts.secure) {
                cookie += '; secure';
            }
        }
        document.cookie = cookie;
    },
    /**
     * Clears a cookie depending on its name.
     * 
     * @param {String} name The name of the cookie to get.
     */
    clear : function(name) {
        document.cookie = name + '=; expires=' + new Date(0).toGMTString(); 
    },
    /**
     * Returns all cookies.
     * 
     * @returns {Object} The cookies.
     */
    all : function() {
        var cookies = document.cookie;
        cookies = cookies.split('; ');
        var all = {};
        for (var i = 0; i < cookies.length; i++) {
            var kv = cookies[i].split('=');
            all[kv[0]] = kv[1];
        }
        return all;
    }
});

return Cookie;
});
